//
//  Controller.m
//  BwanaDik
//
//  Created by John Schilling on 11/28/04.
//  Copyright 2004 John Schilling. All rights reserved.
//

#import "Controller.h"
#import "WANController.h"
#import "LANController.h"

#import "NSStringExtras.h"

#import <Foundation/Foundation.h>
#include <CoreFoundation/CoreFoundation.h>
#include <SystemConfiguration/SystemConfiguration.h>

#define TIMER_SECONDS       15.0
#define UPDATE_RATE_MAX     5

@implementation Controller


#pragma mark
#pragma mark PRIVATE INSTANCE METHODS
#pragma mark

- (id)init
{
    if( ![super init] ) return nil;
    
    _lastKnownWANIPAddress  = [[NSMutableString alloc] init];
    
    _lastUpdatedString      = [[NSMutableString alloc] init];
    _nextUpdateString       = [[NSMutableString alloc] init];
    _lastAppPath            = [[NSMutableString alloc] init];
    _emailToAddress         = [[NSMutableString alloc] init];

    _interfaceMenuItems     = [[NSMutableArray alloc] init];
                
    return self;
}

- (void)awakeFromNib
{
    [self loadStringsFromNib];
    
    _appNameString           = [[[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleExecutable"] retain];
    _versionString           = [[[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleVersion"] retain];
    [_aboutBoxenVersionNumberField setStringValue:[NSString stringWithFormat:@"Version Number %@", _versionString]];
    
    [_alertChangeToggleButton setTarget: self];

    [self loadPreferences];
    [self setupPrefsPanel];
    
    [_WAN setLastKnownIPAddress:_lastKnownWANIPAddress downloadTimeout:_timeoutInterval IPSourceTag:_WANIPsourceTag];
    
    [self initStatusItem];
    [self initInterface];
    
    [self beginCheckingWANaddress:nil];
    [self beginCheckingLANaddress:nil];
}

- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
    
}

- (void)applicationWillTerminate:(NSNotification *)aNotification
{
    [_WAN cleanup];
    [self stopUpdateTimer];
    [self savePreferences];
}

- (void)applicationDidHide:(NSNotification *)aNotification
{
    /* NOP */
}

- (void)mouseDown:(NSEvent *)theEvent
{
    [[self window] makeFirstResponder:[[self window] contentView]];
}

- (BOOL)windowShouldClose:(id)sender
{
    // Prefs window. We need to set the email address from here.
    [self setEmailToAddress: nil];
    [self returnFocusToLastApplication];
    return YES;
}

- (void)dealloc
{
    [_lastKnownWANIPAddress release];
    
    [_appStrings release];
    [_emailStrings release];
    
    [_emailToAddress release];
    _emailToAddress = nil;
    
    [_statusItem release];
    _statusItem = nil;
    
    [_lastUpdatedDate release];
    _lastUpdatedDate = nil;
    [_lastUpdatedString release];
    _lastUpdatedString = nil;
    [_nextUpdateDate release];
    _nextUpdateDate = nil;
    [_nextUpdateString release];
    _nextUpdateString = nil;
    
    [_appNameString release];
    _appNameString = nil;
    [_versionString release];
    _versionString = nil;
    
    [_lastAppPath release];
    _lastAppPath = nil;
    
    [_interfaceMenuItems removeAllObjects];
    [_interfaceMenuItems release];
    
    [super dealloc];
}


#pragma mark
#pragma mark MISC INIT METHODS
#pragma mark

- (void)loadStringsFromNib
{
    NSString     *stringsDataPath   = [[NSBundle mainBundle] pathForResource:@"Localizable" ofType:@"strings" inDirectory: nil];
    NSDictionary *stringsMasterDict = [NSDictionary dictionaryWithContentsOfFile: stringsDataPath];
    
    _appStrings     = [[stringsMasterDict objectForKey:@"ApplicationStrings"] retain];
    if (_appStrings == NULL) {
        NSLog(@"ApplicationStrings failed to load. Application must terminate now.");
        [NSApp terminate:self];
    }

    _emailStrings   = [[stringsMasterDict objectForKey:@"EmailStrings"] retain];
    if (_emailStrings == NULL) {
        NSLog(@"EmailStrings failed to load. Application must terminate now.");
        [NSApp terminate:self];
    }
    
}

- (void)initStatusItem
{
    _statusItem = [[[NSStatusBar systemStatusBar] statusItemWithLength:22.0] retain];
    [_statusItem setHighlightMode:YES];
    [_statusItem setEnabled:YES];
    [_statusItem setMenu:_bwanaMenu];
}

- (void)initInterface
{
    int idx;
    
    [self updateNetworkLocationsMenu];
    
    [_interfaceMenu  setTitle:[self appStringForKey:@"Interfaces"]];
    
    for( idx = 0; idx < [_interfaceMenu numberOfItems]; ++idx )
    {
        [_interfaceMenuItems addObject: [_interfaceMenu itemAtIndex: idx]];
    }


    [self setWANInterfaceItems];
    [self setLANInterfaceItems];
    
    [self setAutoUpdateItems];
}











#pragma mark
#pragma mark PREFERENCE METHODS
#pragma mark

- (void)loadPreferences
{
    [[NSUserDefaults standardUserDefaults] registerDefaults:
     [NSDictionary dictionaryWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"Defaults" ofType:@"plist"]]];
    
    _WANIPsourceTag         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_WANIPsourceTag"];
    
    
    _autoUpdates            = [[NSUserDefaults standardUserDefaults] integerForKey:@"_autoUpdates"];
    _superWANCheck          = [[NSUserDefaults standardUserDefaults] integerForKey:@"_superWANCheck"];
    _updateOnLocalChange    = [[NSUserDefaults standardUserDefaults] integerForKey:@"_updateOnLocalChange"];
    _updateRate             = [[NSUserDefaults standardUserDefaults] integerForKey:@"_updateRate"];
    
    _timeoutInterval        = [[NSUserDefaults standardUserDefaults] integerForKey:@"_timeoutInterval"];
    
    _alertOnAvailable       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertOnAvailable"];
    _alertOnDisconnect      = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertOnDisconnect"];
    _alertOnLANChange       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertOnLANChange"];
    _alertOnWANChange       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertOnWANChange"];
    
    _emailOnAvailable       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_emailOnAvailable"];
    _emailOnLANChange       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_emailOnLANChange"];
    _emailOnWANChange       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_emailOnWANChange"];
    
    [_emailToAddress        setString: [[NSUserDefaults standardUserDefaults] objectForKey:@"_emailToAddress"]];
    
    [_lastKnownWANIPAddress setString: [[NSUserDefaults standardUserDefaults] objectForKey:@"_lastKnownWANIPAddress"]];
    
    _alertPanelAutoCloses   = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertPanelAutoCloses"];
    _iconStyleID            = [[NSUserDefaults standardUserDefaults] integerForKey:@"_iconStyleID"];	
    
    if (_updateRate < 0 || _updateRate > UPDATE_RATE_MAX) _updateRate = 2;
}

- (void)savePreferences
{
    [[NSUserDefaults standardUserDefaults] setInteger:_WANIPsourceTag		forKey:@"_WANIPsourceTag"];
    
    
    [[NSUserDefaults standardUserDefaults] setInteger:_autoUpdates              forKey:@"_autoUpdates"];
    [[NSUserDefaults standardUserDefaults] setInteger:_superWANCheck            forKey:@"_superWANCheck"];
    [[NSUserDefaults standardUserDefaults] setInteger:_updateOnLocalChange	forKey:@"_updateOnLocalChange"];
    [[NSUserDefaults standardUserDefaults] setInteger:_updateRate               forKey:@"_updateRate"];
    
    [[NSUserDefaults standardUserDefaults] setInteger:_timeoutInterval		forKey:@"_timeoutInterval"];
    
    [[NSUserDefaults standardUserDefaults] setInteger:_alertOnAvailable		forKey:@"_alertOnAvailable"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertOnDisconnect        forKey:@"_alertOnDisconnect"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertOnLANChange         forKey:@"_alertOnLANChange"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertOnWANChange         forKey:@"_alertOnWANChange"];
    
    [[NSUserDefaults standardUserDefaults] setInteger:_emailOnAvailable		forKey:@"_emailOnAvailable"];
    [[NSUserDefaults standardUserDefaults] setInteger:_emailOnLANChange		forKey:@"_emailOnLANChange"];
    [[NSUserDefaults standardUserDefaults] setInteger:_emailOnWANChange         forKey:@"_emailOnWANChange"];
    
    [[NSUserDefaults standardUserDefaults] setObject:_emailToAddress            forKey:@"_emailToAddress"];
    
    [[NSUserDefaults standardUserDefaults] setInteger:_alertPanelAutoCloses     forKey:@"_alertPanelAutoCloses"];
    [[NSUserDefaults standardUserDefaults] setInteger:_iconStyleID              forKey:@"_iconStyleID"];
    
    [[NSUserDefaults standardUserDefaults] setObject:[_WAN currentWANAddress]   forKey:@"_lastKnownWANIPAddress"];
    
    [[NSUserDefaults standardUserDefaults] synchronize];
}

- (void)setupPrefsPanel
{
    //autoUpdates is set by setAutoUpdateItems method below

    [_IPSourceTagMenu                           selectItemAtIndex:_WANIPsourceTag];

    [_superWANCheckButton                       setState:(_superWANCheck ? NSOnState : NSOffState)];
    [_updateOnLocalChangeButton                 setState:(_updateOnLocalChange ? NSOnState : NSOffState)];
    [_updateRateMenu				selectItemAtIndex:_updateRate];
    [_updateRateMenu				setEnabled:_autoUpdates];
    
    [_timeoutTextField				setIntValue:_timeoutInterval];
    [_timeoutStepper				setIntValue:_timeoutInterval];
    
    [_alertOnAvailableButton                    setState:_alertOnAvailable];
    [_alertOnDisconnectButton                   setState:_alertOnDisconnect];
    [_alertOnWANChangeButton                    setState:_alertOnWANChange];
    [_alertOnLANChangeButton                    setState:_alertOnLANChange];
    [_emailOnAvailableButton                    setState:_emailOnAvailable];
    [_emailOnLANChangeButton                    setState:_emailOnLANChange];
    [_emailOnWANChangeButton                    setState:_emailOnWANChange];
    [_alertPanelAutoClosesButton                setState:_alertPanelAutoCloses];
    [_emailToAddressField                       setStringValue: _emailToAddress];
    
    [_menuIconStyleMenu				selectItemAtIndex: _iconStyleID];
    
    [_loginItemButton				setState:([self isLoginItem] ? NSOnState : NSOffState)];
}

- (IBAction)openPrefsPanel:(id)sender
{
    [self setLastApplicationPath];
    [NSApp activateIgnoringOtherApps:YES];
    [_prefsPanel center];
    [_prefsPanel makeKeyAndOrderFront:self];
    [_prefsPanel display];
}

- (IBAction)setWANIPSource:(id)sender
{
    _WANIPsourceTag = [[sender selectedItem] tag];
    [_WAN setIPSourceTag:_WANIPsourceTag];
}

- (IBAction)setTimeoutInterval:(id)sender
{
    _timeoutInterval = [sender intValue];
    [_timeoutTextField setIntValue:_timeoutInterval];
}

- (IBAction)setAutoUpdates:(id)sender
{
    _autoUpdates = [sender state];
    [self setAutoUpdateItems];
}

- (IBAction)setSuperWANCheck:(id)sender
{
    _superWANCheck = [sender state];
}

- (IBAction)setUpdateOnLocalChange:(id)sender
{
    _updateOnLocalChange = [sender state];
}

- (IBAction)toggleAutoUpdates:(id)sender
{
    _autoUpdates = !_autoUpdates;
    [self setAutoUpdateItems];
}

- (IBAction)setUpdateRate:(id)sender
{
    _updateRate = [[sender selectedItem] tag];
    [self calculateNextWANUpdate];
    [self checkWANAutoUpdate];
}

- (IBAction)setAlertOnAvailable:(id)sender
{
    _alertOnAvailable = [sender state];
}

- (IBAction)setAlertOnDisconnect:(id)sender
{
    _alertOnDisconnect = [sender state];
}

- (IBAction)setAlertOnWANChange:(id)sender
{
    _alertOnWANChange = [sender state];
}

- (IBAction)setAlertOnLANChange:(id)sender
{
    _alertOnLANChange = [sender state];
}

- (IBAction)setEmailOnAvailable:(id)sender
{
    _emailOnAvailable = [sender state];
}

- (IBAction)setEmailOnLANChange:(id)sender
{
    _emailOnLANChange = [sender state];
}

- (IBAction)setEmailOnWANChange:(id)sender
{
    _emailOnWANChange = [sender state];
}

- (IBAction)reverseAlertOnAvailable:(id)sender
{
    _alertOnAvailable   = ![sender state];
    [_alertOnAvailableButton setState:_alertOnAvailable];
}

- (IBAction)reverseAlertOnDisconnect:(id)sender
{
    _alertOnDisconnect   = ![sender state];
    [_alertOnDisconnectButton setState:_alertOnDisconnect];
}

- (IBAction)reverseAlertOnWANChange:(id)sender
{
    _alertOnWANChange      = ![sender state];
    [_alertOnWANChangeButton setState:_alertOnWANChange];
}

- (IBAction)reverseAlertOnLANChange:(id)sender
{
    _alertOnLANChange      = ![sender state];
    [_alertOnLANChangeButton setState:_alertOnLANChange];
}

- (IBAction)setAutoCloseAlertPanel:(id)sender
{
    _alertPanelAutoCloses = [sender state];
}

- (IBAction)setEmailToAddress:(id)sender
{
    [_emailToAddress setString: [_emailToAddressField stringValue]];
}

- (IBAction)setMenuIconStyle:(id)sender
{
    _iconStyleID = [[sender selectedItem] tag];
    [self setNetworkStatusIcon];
}

- (IBAction)addOrRemoveToLoginItems:(id)sender
{
    if ([sender state]) {
        if (![self isLoginItem]) [self setLoginItem];
    } else {
        if ([self isLoginItem]) [self removeLoginItem];
    }
}

- (void)setAutoUpdateItems
{
    if (_autoUpdates) {
        [_autoUpdatesMenuItem setTitle:[self appStringForKey:@"AutoUpdateOff"]];
        [self checkWANAutoUpdate];
        [self startUpdateTimer];
    } else {
        [self stopUpdateTimer];
        [_autoUpdatesMenuItem setTitle:[self appStringForKey:@"AutoUpdateOn"]];
    }
    
    [_autoUpdatesButton setState:_autoUpdates];
    [_superWANCheckButton setEnabled:_autoUpdates];
    [_updateRateMenu setEnabled:_autoUpdates];
}







































#pragma mark
#pragma mark REFRESH/UPDATE/CANCEL
#pragma mark

- (IBAction)refreshOrCancelUpdate:(id)sender
{
    [self beginCheckingLANaddress:self];
    [_WAN checkOrCancel:self];
}



#pragma mark
#pragma mark WAN METHODS

#pragma mark delegate

- (void)WANCheckBegun
{
    [self setWANInterfaceItems];
}

- (void)WANCheckCanceled
{
    [self setLastWANUpdate];
    [self setWANInterfaceItems];
}

- (void)WANCheckFinished
{
    [self setLastWANUpdate];
    [self setWANInterfaceItems];
}

- (void)WANWentOnline
{
    // NO NEED TO UPDATE INTERFACE, AS WANCheckFinished HAS ALREADY BEEN CALLED
    [self runAlertUp];
}

- (void)WANWentOffline
{
    // NO NEED TO UPDATE INTERFACE, AS WANCheckFinished HAS ALREADY BEEN CALLED
    [self runAlertDown];
}

- (void)WANAddressChangedFrom:(NSString *)previousAddress to:(NSString *)currentAddress
{
    // NO NEED TO UPDATE INTERFACE, AS WANCheckFinished HAS ALREADY BEEN CALLED
    [self runAlertChange:previousAddress new:currentAddress type:1];
}

#pragma mark check

- (IBAction)beginCheckingWANaddress:(id)sender
{
    [_WAN beginChecking:self];
}

- (IBAction)cancelCheckingWANaddress:(id)sender
{
    [_WAN cancelChecking:self];
}

#pragma mark interface

- (void)setWANInterfaceItems
{
    [self setRefreshOrCancelMenuItem];
    [self setNetworkStatusMenuItemTitle];
    [self setNetworkStatusIcon];
    [self setWANaddressMenuItem];
}

- (void)setRefreshOrCancelMenuItem
{
    if ([_WAN isChecking]) {
        [_refreshOrCancelMenuItem setTitle:[self appStringForKey:@"Cancel"]];
    } else {
        [_refreshOrCancelMenuItem setTitle:[self appStringForKey:@"Refresh"]];
    }
}

- (void)setNetworkStatusMenuItemTitle
{
    if ([_WAN isChecking]) {
        [_networkStatusMenuItem setTitle:[self appStringForKey:@"Checking"]];
    } else if ([_WAN currentOnlineStatus] == BDUnknown) {
        [_networkStatusMenuItem setTitle:[self appStringForKey:@"NetworkOffline"]];
    } else if ([_WAN currentOnlineStatus] == BDConnected || [_WAN currentOnlineStatus] == BDBusyServer) {
        [_networkStatusMenuItem setTitle:[self appStringForKey:@"NetworkOnline"]];
    } else if ([_WAN currentOnlineStatus] == BDDisconnected) {
        [_networkStatusMenuItem setTitle:[self appStringForKey:@"NetworkUnavailable"]];
    }
}

- (void)setNetworkStatusIcon
{
    if ([_WAN isChecking]) {
        [_statusItem setImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-3", _iconStyleID]]];
        [_statusItem setAlternateImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-3x", _iconStyleID]]];
    } else {
        if ([_WAN currentOnlineStatus] == BDUnchecked || [_WAN currentOnlineStatus] == BDUnknown)
        {
            [_statusItem setImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-0", _iconStyleID]]];
            [_statusItem setAlternateImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-0x", _iconStyleID]]];
        } else if ([_WAN currentOnlineStatus] == BDBusyServer) {
            [_statusItem setImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-2", _iconStyleID]]];
            [_statusItem setAlternateImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-2x", _iconStyleID]]];
        } else if ([_WAN currentOnlineStatus] == BDDisconnected) {
            [_statusItem setImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-9", _iconStyleID]]];
            [_statusItem setAlternateImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-9x", _iconStyleID]]];
        } else if ([_WAN currentOnlineStatus] == BDConnected) {
            [_statusItem setImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-1", _iconStyleID]]];
            [_statusItem setAlternateImage:[NSImage imageNamed:[NSString stringWithFormat:@"MenuIcon%d-1x", _iconStyleID]]];
        }
    }
}

- (void)setWANaddressMenuItem
{
    if ([_WAN isChecking]) return;
    
    if ([[_WAN currentWANAddress] isValidAddress])
    {
        [_currentWANaddressMenuItem setTitle:[NSString stringWithFormat:@"%@: %@", [self appStringForKey:@"ExternalIP"], [_WAN currentWANAddress]]];
    } else {
        if ([_WAN currentOnlineStatus] == BDDisconnected) {
            [_currentWANaddressMenuItem setTitle:[NSString stringWithFormat:@"%@: %@", 
                                                    [self appStringForKey:@"ExternalIP"], 
                                                    [self appStringForKey:@"UnknownDisconnected"]]];
        } else if ([_WAN currentOnlineStatus] == BDBusyServer) {
            [_currentWANaddressMenuItem setTitle:[NSString stringWithFormat:@"%@: %@", 
                                                    [self appStringForKey:@"ExternalIP"], 
                                                    [self appStringForKey:@"UnknownBusy"]]];
        } else {
            [_currentWANaddressMenuItem setTitle:[NSString stringWithFormat:@"%@: %@", 
                                                    [self appStringForKey:@"ExternalIP"], 
                                                    [self appStringForKey:@"Unknown"]]];
        }
    }
}

#pragma mark autocheck methods

- (int)nextWANUpdateInSeconds
{
    //_updateRate
    int secs = 10 * 60;  // Default, 10 minutes.
    
    switch (_updateRate) {
        
        case 0: // 1 minute
        secs = 60;
        break;
        
        case 1: // 5 minutes
        secs = 5 * 60;
        break;
        
        case 2: // 10 minutes
        secs = 10 * 60;
        break;
        
        case 3: // 15 minutes
        secs = 15 * 60;
        break;
        
        case 4: // 30 minutes
        secs = 30 * 60;
        break;
        
        case 5: // 1 hour
        secs = 60 * 60;
        break;
    
        default: //10 minutes
        secs = 10 * 60;
        break;
    
    }
    return secs;
}

- (void)calculateNextWANUpdate
{
    if (_nextUpdateDate) [_nextUpdateDate release];
    _nextUpdateDate = nil;
    NSCalendarDate *nextDate = [_lastUpdatedDate dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:[self nextWANUpdateInSeconds]];
    [_nextUpdateString setString:[nextDate descriptionWithCalendarFormat:@"%m/%d %I:%M%p"]];
    _nextUpdateDate = [nextDate retain];
}   

- (void)setLastWANUpdate
{
    if (_lastUpdatedDate) [_lastUpdatedDate release];
    _lastUpdatedDate = nil;
    _lastUpdatedDate = [[NSCalendarDate calendarDate] retain];
    [_lastUpdatedString setString:[_lastUpdatedDate descriptionWithCalendarFormat:@"%m/%d %I:%M%p"]];
    [self calculateNextWANUpdate];
}








#pragma mark
#pragma mark LAN METHODS

#pragma mark delegate

- (void)LANCheckFinished
{
    [self setLANInterfaceItems];
}

- (void)LANAddressChangedFrom:(NSString *)previousAddress to:(NSString *)currentAddress
{
    if (_updateOnLocalChange) {
        [self beginCheckingWANaddress:self];
    }
    [self runAlertChange:previousAddress new:currentAddress type:0];
    
}

#pragma mark check

- (IBAction)beginCheckingLANaddress:(id)sender
{
    [_LAN getCurrentLANAddresses];
}

#pragma mark interface

- (void)setLANInterfaceItems
{
    [self setLANaddressMenuItemTitle];
    [self setInterfacesMenuItemTitles];
}

-(void)setLANaddressMenuItemTitle
{
    if ([[_LAN currentLANAddress] isValidAddress]) {
        [_currentLANaddressMenuItem setTitle:[NSString stringWithFormat:@"%@: %@", [self appStringForKey:@"LocalIP"], [_LAN currentLANAddress]]];
    } else {
        [_currentLANaddressMenuItem setTitle:[NSString stringWithFormat:@"%@: %@", 
                                                [self appStringForKey:@"LocalIP"], 
                                                [self appStringForKey:@"Unknown"]]];
    }
}

- (void)setInterfacesMenuItemTitles
{
    int         idx;
    NSString   *pInterfaceName;
    NSString   *pAddress;
    NSMenuItem *pMenuItem;
   

    /*
    ** Remove all interface menu items from the Interfaces submenu so that if we
    ** do not have enough interfaces to use up all the menu items, then the
    ** unused ones will not be visible.
    */
    for( idx = [_interfaceMenu numberOfItems] - 1; idx >= 0; --idx )
    {
        [_interfaceMenu removeItemAtIndex: 0];
    }
    
    /*
    ** Build up each interface's menu item.
    */
    for( idx = 0; (idx < [_LAN numWiredInterfaces] + 2) && (idx < [_interfaceMenuItems count]); ++idx )
    {
        /*
        ** Grab the next available menu item from the array.
        */
        pMenuItem = [_interfaceMenuItems objectAtIndex: idx];
        
        /*
        ** Find out what the address is for this interface.
        */
        pAddress = [_LAN interfaceAddressForIndex: idx];
        if( ![pAddress isValidAddress] )
        {
            /*
            ** If the address is not valid, then change it's text as appropriate.
            */
            pAddress = [self appStringForKey: @"None"];
        }
        
        /*
        ** Figure out what the interface type is so that it can be labelled
        ** correctly.
        */
        if( idx == [_LAN numWiredInterfaces] )
        {
            pInterfaceName = [NSString stringWithFormat: @"Airport  (en%d)", idx];
        }
        else if( idx == ([_LAN numWiredInterfaces] + 1) )
        {
            pInterfaceName = @"Firewire (fw0)";
        }
        else if( (idx >= 0) && (idx < [_LAN numWiredInterfaces]) )
        {
            pInterfaceName = [NSString stringWithFormat: @"Ethernet (en%d)", idx];
        }
        else
        {
            pInterfaceName = @"Interface";
        }
        
        /*
        ** Because this interface item is in use, enable the menu item and then
        ** set it's title to what we just built up.
        */
        [_interfaceMenu addItem: pMenuItem];
        [pMenuItem setTitle: [NSString stringWithFormat: @"%@: %@",
                             pInterfaceName, 
                             pAddress]];
    }
}









#pragma mark
#pragma mark LOCATION METHODS
#pragma mark

- (NSDictionary *)getLocationNames
{
    NSMutableDictionary *locations = [[NSMutableDictionary alloc] init];
    
    SCPreferencesRef prefRef = SCPreferencesCreate(kCFAllocatorSystemDefault, (CFStringRef)@"XXX", NULL);
    NSDictionary *allSets = (NSDictionary *)SCPreferencesGetValue(prefRef, kSCPrefSets);
    NSEnumerator *setEnumerator = [allSets keyEnumerator];
    NSString *setId;

    while(setId = [setEnumerator nextObject]) {
        [locations setObject:[[allSets objectForKey:setId] objectForKey:(NSString *)kSCPropUserDefinedName] forKey:setId];
    }
    
    return [locations autorelease];
}

- (NSString *)getCurrentLocationName
{
    SCPreferencesRef prefRef = SCPreferencesCreate(kCFAllocatorSystemDefault, (CFStringRef)@"XXX", NULL);
    NSString *setPath = (NSString *)SCPreferencesGetValue(prefRef, kSCPrefCurrentSet);
    return [setPath lastPathComponent];
}

- (void)updateNetworkLocationsMenu
{
    [_networkLocationsMenuItem setSubmenu:nil];
    
    NSString *currentLoc = [self getCurrentLocationName];
    
    NSDictionary *locs = [self getLocationNames];
    if ([locs count] < 1) return;
    NSMenu *newMenu = [[[NSMenu alloc] init] autorelease];
    
    NSArray *sortedKeyArray = [locs keysSortedByValueUsingSelector:@selector(compare:)];
    NSEnumerator *locationsEnum = [sortedKeyArray objectEnumerator];
    NSString *key;
    while(key = [locationsEnum nextObject]) {
        NSString *title = [locs objectForKey: key];
        NSMenuItem *mi = [newMenu addItemWithTitle:title action:@selector(setLocation:) keyEquivalent:@""];
        [mi setRepresentedObject: key];
        [mi setTarget:self];
        if (currentLoc && [currentLoc isEqualToString: key]) [mi setState:NSOnState];
    }
    [_networkLocationsMenuItem setSubmenu:newMenu];
}













#pragma mark
#pragma mark USER ACTION METHODS
#pragma mark


- (IBAction)copyWANAddressToClipboard:(id)sender
{
    NSString *pAddress = [_WAN currentWANAddress];
    if( ![pAddress isValidAddress])
    {
        pAddress = @"";
    }
    [[NSPasteboard generalPasteboard] declareTypes: [NSArray arrayWithObject: NSStringPboardType] owner:nil];
    [[NSPasteboard generalPasteboard] setString: pAddress forType:NSStringPboardType];
}

- (IBAction)copyLANAddressToClipboard:(id)sender
{
    NSString *pAddress = [_LAN currentLANAddress];
    if( ![pAddress isValidAddress])
    {
        pAddress = @"";
    }
    [[NSPasteboard generalPasteboard] declareTypes: [NSArray arrayWithObject: NSStringPboardType] owner:nil];
    [[NSPasteboard generalPasteboard] setString: pAddress forType:NSStringPboardType];
}

- (IBAction)copyInterfaceToClipboard:(id)sender
{
    NSString *interfaceAddress = [_LAN interfaceAddressForIndex:[sender tag]];
    if( ![interfaceAddress isValidAddress])
    {
        interfaceAddress = @"";
    }
    [[NSPasteboard generalPasteboard] declareTypes:[NSArray arrayWithObject: NSStringPboardType] owner:nil];
    [[NSPasteboard generalPasteboard] setString:interfaceAddress forType:NSStringPboardType];
}

- (IBAction)setLocation:(id)sender
{
    NSString *aeString = [NSString stringWithFormat:@"set theResult to do shell script (\" scselect %@ \") as string", [sender representedObject]];
  	
    NSString *result = [[[[[NSAppleScript alloc] initWithSource:aeString] autorelease] executeAndReturnError:nil] stringValue];

    [self updateNetworkLocationsMenu];
}

- (IBAction)openNetworkPrefPane:(id)sender
{
    [NSTask launchedTaskWithLaunchPath:@"/usr/bin/open" arguments:[NSArray arrayWithObject:@"/System/Library/PreferencePanes/Network.prefPane"]];
}




















#pragma mark
#pragma mark AUTOUPDATE TIMER
#pragma mark

- (void)stopUpdateTimer
{
    if (_updateTimer) [_updateTimer invalidate];
    if (_updateTimer) [_updateTimer release];
    _updateTimer = nil;
}

- (void)startUpdateTimer
{
    [self stopUpdateTimer]; 
        
    NSTimer *timer = [NSTimer scheduledTimerWithTimeInterval:TIMER_SECONDS
                            target:self 
                            selector:@selector(updateTimerFired:) 
                            userInfo:nil 
                            repeats:YES];
    if (timer) _updateTimer = [timer retain];
}

- (void)updateTimerFired:(NSTimer *)timer
{
    [self beginCheckingLANaddress:self];
    [self checkWANAutoUpdate];
}

- (void)checkWANAutoUpdate
{
    if ([_WAN isChecking]) return;
    
    if (!_autoUpdates) return;
    
    if (_superWANCheck && [_WAN currentOnlineStatus] == BDDisconnected)
    {
        [self beginCheckingWANaddress:nil];
        return;
    }
    
    NSCalendarDate *now = [NSCalendarDate calendarDate];
    
    if (_nextUpdateDate) {
        if ([_nextUpdateDate isEqualToDate:now] || [now isEqualToDate:[now laterDate:_nextUpdateDate]]) {
            [self beginCheckingWANaddress:nil];
        } else {
            return;
        }
    } else {
        [self beginCheckingWANaddress:nil];
    }
}















#pragma mark
#pragma mark ALERTS
#pragma mark


- (void)runAlertUp
{
    if (_alertOnAvailable) {
        [_alerterDown killAlert];
        [_alerterChange killAlert];
        [_alerterUp runAlertPanelWithWANAddress:[_WAN currentWANAddress] 
                                     LANAddress:[_LAN currentLANAddress] 
                                         status:[_WAN currentOnlineStatus]
                                      autoClose:_alertPanelAutoCloses];
    }
    if (_emailOnAvailable)
    {
        //[self sendEmailForAvailable];
    }
}

- (void)runAlertDown
{
    if (_alertOnDisconnect) {
        [_alerterUp killAlert];
        [_alerterChange killAlert];
        [_alerterDown runAlertPanelWithStatus:[_LAN currentLANAddress] 
                                       status:[_WAN currentOnlineStatus] 
                                    autoClose:_alertPanelAutoCloses];
    }
}

- (void)runAlertChange:(NSString *)oldIP new:(NSString *)newIP type:(int)type
{
    if ([oldIP isValidAddress] && [newIP isValidAddress])
    {
        if ((_alertOnLANChange && type == 0) || (_alertOnWANChange && type == 1))
        {
            if (type == 0) {
                [_alertChangeToggleButton setAction:@selector(reverseAlertOnLANChange:)];
            } else {
                [_alertChangeToggleButton setAction:@selector(reverseAlertOnWANChange:)];
            }
            [_alerterUp killAlert];
            [_alerterDown killAlert];
            [_alerterChange runAlertPanelChangedAddress:newIP 
                                        PreviousAddress:oldIP 
                                             changeType:type
                                              autoClose:_alertPanelAutoCloses];
        }
        if ((_emailOnLANChange && type == 0) || (_emailOnWANChange && type == 1))
        {
            if (type == 0) {
                //[self sendEmailForLANAddressChange:newIP oldIP:oldIP];
            } else {
                //[self sendEmailForWANAddressChange:newIP oldIP:oldIP];
            }
        }
    }
}



#pragma mark
#pragma mark EMAILING METHODS
#pragma mark

- (void)sendEmailForAvailable
{
    if (![_emailToAddress isValidEmail]) {
        NSLog(@"Specified email address for alerts is invalid (%@)", _emailToAddress);
        return;
    }
    NSString *messageString =  [self formatEmailMessage:[_emailStrings objectForKey:@"EmailAvailableBody"]
                                        currentLAN:[_LAN currentLANAddress]
                                        currentWAN:[_WAN currentWANAddress]
                                        previousLAN:[_LAN previousLANAddress]
                                        previousWAN:[_WAN previousWANAddress]];
    NSString *headerString  =  [self formatEmailMessage:[_emailStrings objectForKey:@"EmailAvailableSubject"]
                                        currentLAN:[_LAN currentLANAddress]
                                        currentWAN:[_WAN currentWANAddress]
                                        previousLAN:[_LAN previousLANAddress]
                                        previousWAN:[_WAN previousWANAddress]];
                                                                                                          
    NSAttributedString *message = [[[NSAttributedString alloc] initWithString:messageString] autorelease];

    NSMutableDictionary *headers = [[NSMutableDictionary alloc] init]; 
    [headers setObject:_emailToAddress forKey:@"To"];
    [headers setObject: headerString forKey:@"Subject"];

    id emailer = [[[EmailThread alloc] init] autorelease];
    [NSThread detachNewThreadSelector:@selector(sendEmail:) toTarget:emailer 
        withObject:[NSArray arrayWithObjects:headers , message, nil]];
    [headers release];
}

- (void)sendEmailForLANAddressChange:(NSString *)newIP oldIP:(NSString *)oldIP
{
    if (![_emailToAddress isValidEmail]) {
        NSLog(@"Specified email address for alerts is invalid (%@)", _emailToAddress);
        return;
    }
    NSString *messageString =  [self formatEmailMessage:[_emailStrings objectForKey:@"EmailChangeLANBody"]
                                        currentLAN: newIP
                                        currentWAN: [_WAN currentWANAddress]
                                        previousLAN: oldIP
                                        previousWAN: [_WAN previousWANAddress]];
    NSString *headerString  =  [self formatEmailMessage:[_emailStrings objectForKey:@"EmailChangeLANSubject"]
                                        currentLAN: newIP
                                        currentWAN: [_WAN currentWANAddress]
                                        previousLAN: oldIP
                                        previousWAN: [_WAN previousWANAddress]];
                                                                                                          
    NSAttributedString *message = [[[NSAttributedString alloc] initWithString:messageString] autorelease];

    NSMutableDictionary *headers = [[NSMutableDictionary alloc] init]; 
    [headers setObject:_emailToAddress forKey:@"To"];
    [headers setObject: headerString forKey:@"Subject"];

    id emailer = [[[EmailThread alloc] init] autorelease];
    [NSThread detachNewThreadSelector:@selector(sendEmail:) toTarget:emailer 
        withObject:[NSArray arrayWithObjects:headers , message, nil]];
    [headers release];
}

- (void)sendEmailForWANAddressChange:(NSString *)newIP oldIP:(NSString *)oldIP
{
    if (![_emailToAddress isValidEmail]) {
        NSLog(@"Specified email address for alerts is invalid (%@)", _emailToAddress);
        return;
    }
    NSString *messageString =  [self formatEmailMessage:[_emailStrings objectForKey:@"EmailChangeWANBody"]
                                        currentLAN: [_LAN currentLANAddress]
                                        currentWAN: newIP
                                        previousLAN:[_LAN previousLANAddress]
                                        previousWAN: oldIP];
    NSString *headerString  =  [self formatEmailMessage:[_emailStrings objectForKey:@"EmailChangeWANSubject"]
                                        currentLAN: [_LAN currentLANAddress]
                                        currentWAN: newIP
                                        previousLAN:[_LAN previousLANAddress]
                                        previousWAN: oldIP];
                                                                                                          
    NSAttributedString *message = [[[NSAttributedString alloc] initWithString:messageString] autorelease];

    NSMutableDictionary *headers = [[NSMutableDictionary alloc] init]; 
    [headers setObject:_emailToAddress forKey:@"To"];
    [headers setObject: headerString forKey:@"Subject"];

    id emailer = [[[EmailThread alloc] init] autorelease];
    [NSThread detachNewThreadSelector:@selector(sendEmail:) toTarget:emailer 
        withObject:[NSArray arrayWithObjects:headers , message, nil]];
    [headers release];
}

- (NSString *)formatEmailMessage:(NSString *)message 
                currentLAN:(NSString *)currentLAN 
                currentWAN:(NSString *)currentWAN 
               previousLAN:(NSString *)previousLAN 
               previousWAN:(NSString *)previousWAN 
{
    NSMutableString *returnMessage = [[[NSMutableString alloc] initWithString: message] autorelease];
    if (returnMessage && [returnMessage length] > 0) {
        [returnMessage replaceOccurrencesOfString:@"<timestamp>" withString:[[NSCalendarDate calendarDate] descriptionWithCalendarFormat:@"%m/%d at %I:%M%p"] options:nil range:NSMakeRange(0, [returnMessage length])];
        [returnMessage replaceOccurrencesOfString:@"<currentLANIP>" withString:currentLAN options:nil range:NSMakeRange(0, [returnMessage length])];
        [returnMessage replaceOccurrencesOfString:@"<currentWANIP>" withString:currentWAN options:nil range:NSMakeRange(0, [returnMessage length])];
        [returnMessage replaceOccurrencesOfString:@"<previousLANIP>" withString:previousLAN options:nil range:NSMakeRange(0, [returnMessage length])];
        [returnMessage replaceOccurrencesOfString:@"<previousWANIP>" withString:previousWAN options:nil range:NSMakeRange(0, [returnMessage length])];
    }
    return returnMessage;
}



















#pragma mark
#pragma mark ABOUT BOX METHODS
#pragma mark

- (IBAction)openAboutBoxen:(id)sender
{
    [self setLastApplicationPath];
    [NSApp activateIgnoringOtherApps:YES];
    [_aboutBoxen center];
    [_aboutBoxen makeKeyAndOrderFront:self];
    [_aboutBoxen display];
}

- (IBAction)linkClicked:(id)sender
{
    NSURL *url = [NSURL URLWithString: [sender stringValue]];
    if (url == nil) return;
    [[NSWorkspace sharedWorkspace] openURL:url];
    [_aboutBoxen close];
}

- (IBAction)mailLinkClicked:(id)sender
{
    [[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:@"mailto:john@jschilling.net?subject=BwanaDik"]];
    [_aboutBoxen close];
}

- (IBAction)donateLinkClicked:(id)sender
{
    [[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:@"https://www.paypal.com/xclick/business=john%40jschilling.net&no_note=1&tax=0&currency_code=USD"]];
    [_aboutBoxen close];
}






#pragma mark
#pragma mark FOO METHODS
#pragma mark

- (NSString *)appStringForKey:(NSString *)key
{
    if (_appStrings == nil || [_appStrings count] < 1) return @"<Undefined String>";
    if ([_appStrings objectForKey:key])
    {
        return [_appStrings objectForKey:key];
    } else {
        return @"<Undefined String>";
    }
}

- (BOOL)isLoginItem
{
    id obj;

    NSString *ourAppsPath = [[NSBundle mainBundle] bundlePath];
    NSDictionary *loginItemDict = 
        [NSDictionary dictionaryWithContentsOfFile: [NSString stringWithFormat: @"%@/Library/Preferences/loginwindow.plist", NSHomeDirectory()]];
    NSEnumerator *loginItemEnumerator = 
        [[loginItemDict objectForKey:
            @"AutoLaunchedApplicationDictionary"] objectEnumerator];

    while (( obj = [loginItemEnumerator nextObject] ) != nil )
    {
        if ( [[obj objectForKey:@"Path"] isEqualTo:ourAppsPath] )
            return( YES );
    }
    return( NO );
}

- (void)setLoginItem
{
    NSMutableArray *loginItems;
    
    loginItems = (NSMutableArray *)CFPreferencesCopyValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" ,
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    loginItems = [[loginItems autorelease] mutableCopy]; 
    
    NSMutableDictionary *myDict = [[[NSMutableDictionary alloc] init] autorelease];
    [myDict setObject:[NSNumber numberWithBool:NO] forKey:@"Hide"];
    [myDict setObject:[[NSBundle mainBundle] bundlePath] forKey:@"Path"];
    
    [loginItems removeObject:myDict]; //make sure it's not already in there 
    [loginItems addObject:myDict]; 
    
    CFPreferencesSetValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" , 
        loginItems , 
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    CFPreferencesSynchronize(
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    
    [loginItems release];
}

- (void)removeLoginItem
{
    NSMutableArray *loginItems;
    
    loginItems = (NSMutableArray *)CFPreferencesCopyValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" ,
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    loginItems = [[loginItems autorelease] mutableCopy];
    
    NSMutableDictionary *myDict = [[[NSMutableDictionary alloc] init] autorelease];
    [myDict setObject:[NSNumber numberWithBool:NO] forKey:@"Hide"];
    [myDict setObject:[[NSBundle mainBundle] bundlePath] forKey:@"Path"];
    
    [loginItems removeObject:myDict];
    
    CFPreferencesSetValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" , 
        loginItems , 
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    CFPreferencesSynchronize(
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    
    [loginItems release];
}






- (void)setLastApplicationPath
{
    NSDictionary *appDict = [[NSWorkspace sharedWorkspace] activeApplication];
    if (appDict == nil) {
        _lastAppPath = nil;
        return;
    }
    NSString *myPath = [[NSBundle mainBundle] bundlePath];
    NSString *appPath = [appDict objectForKey:@"NSApplicationPath"];
    if (![myPath isEqualToString:appPath]) {
        [_lastAppPath setString: appPath];
    }
}

- (void)returnFocusToLastApplication
{
    if (_lastAppPath == nil || ([_lastAppPath length] < 1)) return;
    NSArray *activeApps = [[NSWorkspace sharedWorkspace] launchedApplications];
    
    NSEnumerator *appsEnum = [activeApps objectEnumerator];
    id anApp;
    while (anApp = [appsEnum nextObject]) {
        NSString *appPath = [anApp objectForKey:@"NSApplicationPath"];
        NSString *appName = [anApp objectForKey:@"NSApplicationName"];
        if ([appPath isEqualToString:_lastAppPath] && ![appName isEqualToString:@"Finder"]) {
            [[NSWorkspace sharedWorkspace] launchApplication: _lastAppPath];
        }
    }
}


@end
