/*
SRIconManager.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRAppController.h"
#import "SRIconManager.h"

// Default name
NSString*   SRIconAquaDefaultName = @"default";
NSString*   SRIconMetalDefaultName = @"metalDefault";

@interface SRIconManager (private)
- (void)_loadDefaultIcon:(int)texture;
@end

#pragma mark -

@implementation SRIconManager

//--------------------------------------------------------------//
#pragma -- Initialize --
//--------------------------------------------------------------//

+ (SRIconManager*)sharedInstance
{
    static SRIconManager*   _sharedInstance = nil;
    if (!_sharedInstance) {
        _sharedInstance = [[SRIconManager alloc] init];
    }
    
    return _sharedInstance;
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize member variables
    _allIconsDict = [[NSMutableDictionary dictionary] retain];
    
    // Load default icons
    [self _loadDefaultIcon:HMWindowTextureAqua];
    [self _loadDefaultIcon:HMWindowTextureMetal];
    
    return self;
}

- (void)dealloc
{
    [_allIconsDict release];
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma -- Icon loading --
//--------------------------------------------------------------//

- (void)_loadDefaultIcon:(int)texture
{
    NSMutableDictionary*    iconDict;
    iconDict = [NSMutableDictionary dictionary];
    
    // Get default icon plist
    NSString*       plistPath = nil;
    NSDictionary*   plist;
    switch (texture) {
    case HMWindowTextureAqua: {
        plistPath = [[NSBundle mainBundle] pathForResource:@"ShiiraIcons" ofType:@"plist"];
        break;
    }
    case HMWindowTextureMetal: {
        plistPath = [[NSBundle mainBundle] pathForResource:@"ShiiraIcons" 
                ofType:@"plist" inDirectory:@"metal"];
        break;
    }
    }
    if (!plistPath) {
        // Could not find plist
        return;
    }
    plist = [NSDictionary dictionaryWithContentsOfFile:plistPath];
    if (!plist) {
        // Invalid plist
        return;
    }
    [iconDict setObject:plist forKey:@"ShiiraIcons"];
    
    // Get icon directory
    NSString*   iconDir;
    iconDir = [plistPath stringByDeletingLastPathComponent];
    
    // Load default icons
    NSDictionary*   iconFileNameDict;
    NSEnumerator*   enumerator;
    NSString*       iconId;
    iconFileNameDict = [plist objectForKey:@"icons"];
    enumerator = [[SRAppController allToolbarItemIdentifiers] objectEnumerator];
    while (iconId = [enumerator nextObject]) {
        // Get file name
        NSString*   fileName;
        fileName = [iconFileNameDict objectForKey:iconId];
        if (fileName) {
            // Load icon image
            NSImage*    iconImage;
            iconImage = [[NSImage alloc] initWithContentsOfFile:
                    [iconDir stringByAppendingPathComponent:fileName]];
            if (iconImage) {
                [iconDict setObject:iconImage forKey:iconId];
                [iconImage release];
            }
        }
    }
    
    // Set icon dictionary root
    switch (texture) {
    case HMWindowTextureAqua: {
        [_allIconsDict setObject:iconDict forKey:SRIconAquaDefaultName];
        break;
    }
    case HMWindowTextureMetal: {
        [_allIconsDict setObject:iconDict forKey:SRIconMetalDefaultName];
        break;
    }
    }
}

- (NSDictionary*)allIconsDict
{
    return _allIconsDict;
}

- (NSDictionary*)iconDictForName:(NSString*)name
{
    NSDictionary*   icons;
    icons = [_allIconsDict objectForKey:name];
    if (!icons) {
        icons = [_allIconsDict objectForKey:SRIconAquaDefaultName];
    }
    
    return icons;
}

@end
