/**********************************************************************
Copyright (c) 2006 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.plugin;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.PersistenceConfiguration;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;

/**
 * Factory for PluginRegistry. 
 * Creates an instance of PluginRegistry based on the available
 * PluginRegistry implementation in the Core classpath
 */
public class PluginRegistryFactory
{
    protected static final Localiser LOCALISER = Localiser.getInstance("org.datanucleus.Localisation",
        ObjectManagerFactoryImpl.class.getClassLoader());

    /**
     * Instantiates a PluginRegistry. If the JPOX Core is deployed as an
     * eclipse plugin, uses the Eclipse OSGI Registry to find other JPOX plug-ins
     * @param conf the PersistenceConfiguration
     * @param clr the ClassLoaderResolver
     * @return instance of the PluginRegistry
     */
    public static PluginRegistry newPluginRegistry(PersistenceConfiguration conf, ClassLoaderResolver clr)
    {
        PluginRegistry registry = null;
        String regClassName = conf.getStringProperty("datanucleus.plugin.pluginRegistryClassName");
        if (regClassName != null)
        {
            registry = newInstance(regClassName, regClassName, clr);
        }
        if (registry != null)
        {
            if (NucleusLogger.PLUGIN.isDebugEnabled())
            {
                NucleusLogger.PLUGIN.debug("Using PluginRegistry " + registry.getClass().getName());
            }
            return registry;
        }
        registry = newInstance("org.eclipse.core.runtime.RegistryFactory", "org.datanucleus.plugin.EclipsePluginRegistry", clr);
        if (registry != null)
        {
            if (NucleusLogger.PLUGIN.isDebugEnabled())
            {
                NucleusLogger.PLUGIN.debug("Using PluginRegistry " + registry.getClass().getName());
            }
            return registry;
        }

        if (NucleusLogger.PLUGIN.isDebugEnabled())
        {
            NucleusLogger.PLUGIN.debug("Using PluginRegistry " + NonManagedPluginRegistry.class.getName());
        }
        return new NonManagedPluginRegistry(clr, conf.getStringProperty("datanucleus.plugin.pluginRegistryBundleCheck"));
    }

    /**
     * Instantiates a PluginRegistry. Only proceed if the testClass is found in
     * the classpath
     * @param testClass A test class
     * @param pluginRegistryClass The class that implements
     * {@link PluginRegistry}
     * @return instance of the PluginRegistry
     */
    private static PluginRegistry newInstance(String testClass, String pluginRegistryClass, ClassLoaderResolver clr)
    {
        try
        {
            if (clr.classForName(testClass, ObjectManagerFactoryImpl.class.getClassLoader()) == null)
            {
                // Just treat all exceptions the same since this registry doesnt work either way
                if (NucleusLogger.PLUGIN.isDebugEnabled())
                {
                    NucleusLogger.PLUGIN.debug(LOCALISER.msg("024005", pluginRegistryClass));
                }
            }
            return (PluginRegistry) clr.classForName(pluginRegistryClass, ObjectManagerFactoryImpl.class.getClassLoader()).getConstructor(
                new Class[] { ClassLoaderResolver.class }).newInstance(new Object[] { clr });
        }
        catch (Exception e)
        {
            // Just treat all exceptions the same since this registry doesnt work either way
            if (NucleusLogger.PLUGIN.isDebugEnabled())
            {
                // Exception.getCause only in JDK1.4+
                NucleusLogger.PLUGIN.debug(LOCALISER.msg("024006", pluginRegistryClass,
                    e.getCause() != null ? e.getCause().getMessage() : e.getMessage()));
            }
        }
        return null;
    }
}