/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Vector;

/**
 * This class provides convenience methods to access a
 * database via jdbc.
 *
 * @author    Chr. Clemens Lee (clemens@kclee.com)
 * @version   $Id: DBUtil.java,v 1.8 2003/05/01 16:44:38 clemens Exp clemens $
 */
public class DBUtil 
{
    /** Constant to indicate an InstantDB database. */
    static public final DBConstant INSTANTDB  = new DBConstant();
    /** Constant to indicate a Sybase database. */
    static public final DBConstant SYBASE     = new DBConstant();
    /** Constant to indicate a PostgreSQL database. */
    static public final DBConstant POSTGRESQL = new DBConstant();

    static private boolean _bInitialized = false;
    static private DBConstant _database = null;

    /**
     * This class is used to indicate the database currently
     * used.
     */
    static public class DBConstant
    {
        private DBConstant() 
{ }
    }   

    /**
     * Initializes an InstantDB database.
     *
     * @exception   Exception   if loading the driver class fails.
     */
    static public void initializeIDB() 
        throws Exception
    {
        initializeDB( "org.enhydra.instantdb.jdbc.idbDriver" );
    }

    /**
     * Initializes a PostgreSQL database.
     *
     * @exception   Exception   if loading the driver class fails.
     */
    static public void initializePGDB() 
        throws Exception
    {
        initializeDB( "org.postgresql.Driver" );
    }

    /**
     * Initializes a Sybase database.
     *
     * @exception   Exception   if loading the driver class fails.
     */
    static public void initializeSybDB() 
        throws Exception
    {
        initializeDB( "com.sybase.jdbc2.jdbc.SybDriver" );
    }

    /**
     * Initializes a database with the given driver.
     *
     * @exception   Exception   if loading the driver class fails.
     */
    static public void initializeDB( String sDriver ) 
        throws Exception
    {
        if ( _bInitialized ) 
{
            return;
        }

        // connect to database
        Class.forName( sDriver ).newInstance();
        _bInitialized = true;
    }

    /**
     * Initializes a database for either InstantDB,
     * Sybase or whatsoever.
     *
     * @exception   Exception   if loading the driver class fails.
     */
    static public void initializeDB( DBConstant database ) 
        throws Exception
    {
        if ( database == INSTANTDB )
        {
            initializeIDB();
        } 
        else if ( database == POSTGRESQL )
        {
            initializePGDB();
        }
        else if ( database == SYBASE )
        {
            initializeSybDB();
        }
        _database = database;
    }

    /**
     * Creates a instantdb database connection.
     *
     * @exception   SQLException   if creating connection fails.
     */
    static public Connection createIDBConnection( String sDatabaseFile )
        throws SQLException
    {
        try
        {
            initializeIDB();
        }
        catch( Exception exception )
        {
            throw new SQLException( "Could not load idb driver: " + exception.getMessage() );
        }

        String sUser     = "no_use";
        String sPassword = "no_password";
        Connection connection = DriverManager.getConnection
               ( "jdbc:idb:" + sDatabaseFile
                 , sUser
                 , sPassword );
                    
        return connection;
    }

    /**
     * Creates a PostgreSQL database connection on the local host 
     * and the current user given the database name.
     *
     * @exception   SQLException   if creating connection fails.
     */
    static public Connection createPGConnection( String sDatabaseName )
        throws SQLException
    {
        try
        {
            initializePGDB();
        }
        catch( Exception exception )
        {
            throw new SQLException( "Could not load postgresql driver: " + exception.getMessage() );
        }

        String sUser     = (String) System.getProperties().get( "user.name" );
        String sPassword = "no_password";
        Connection connection = DriverManager.getConnection
               ( "jdbc:postgresql:" + sDatabaseName
                 , sUser
                 , sPassword );
                    
        return connection;
    }

    /**
     * Creates a database connection.
     *
     * @exception   SQLException   if creating connection fails.
     */
    static public Connection createConnection( DBConstant database
                                               , String url
                                               , String user
                                               , String password
                                               , String applicationname
                                               , String clienthost )
        throws SQLException
    {
        try
        {
            initializeDB( database );
        }
        catch( Exception exception )
        {
            throw new SQLException( "Could not load db driver: " + exception.getMessage() );
        }

        Properties prop = new Properties();
        prop.put( "user", user );
        prop.put( "password", password );
        prop.put( "applicationname", applicationname );
        prop.put( "hostname", clienthost );
        Connection connection = DriverManager.getConnection
               ( url, prop );
                    
        return connection;
    }

    /**
     * Creates a jsbc statement for the given instantdb database.
     *
     * @exception   SQLException   if creating connection fails.
     */
    static public Statement createIDBStatement( String sDatabaseFile ) 
        throws SQLException
    {
        try
        {
            initializeIDB();
        }
        catch( Exception exception )
        {
            throw new SQLException( "Could not load idb driver: " + exception.getMessage() );
        }

        Connection connection = createIDBConnection( sDatabaseFile );

        Statement statement = connection.createStatement();

        return statement;
    }

    /**
     * Closes the given statement and the connection it belongs to.
     *
     * @exception   SQLException   if anything goes wrong.
     */
    static public void close( Statement statement_ ) 
        throws SQLException
    {
        Connection connection = statement_.getConnection();
        statement_.close();
        connection.close();
    }

    /**
     * Deletes the given table.
     *
     * @exception   SQLException   if anything goes wrong.
     */
    static public void deleteTable( Connection connection_
                                    , String sTableName_ ) 
        throws SQLException
    {
        Statement statement = connection_.createStatement();
        
        statement.executeUpdate( "delete from " + sTableName_ );

        statement.close();
    }

    /**
     * Gets the list of InstantDB user tables.
     *
     * @exception   Exception   if anything goes wrong.
     */
    static public Vector getIDBUserTables( String sDatabaseFile ) 
        throws Exception
    {
        initializeIDB();
        Connection connection = createIDBConnection( sDatabaseFile );

        Vector vRetVal = getUserTables( connection );

        connection.close();

        return vRetVal;
    }

    /**
     * Gets the list of user tables.
     *
     * @exception   Exception   if anything goes wrong.
     */
    static public Vector getUserTables( Connection connection )
        throws Exception
    {
        DatabaseMetaData metaData = connection.getMetaData();

        String[] asTableType = { "TABLE" };
        ResultSet result = metaData.getTables( null, null, null, asTableType );
        Vector vResVal = new Vector();
        while( result.next() ) 
	{
            String sTableSchema = result.getString( 2 );
            if ( sTableSchema != null 
                 && sTableSchema.trim().equals( "SYS_INFO" ) )
            {
                // system table from mckoi database - ignore
                continue;
            }
            vResVal.addElement( result.getString( 3 ) );
        }
        result.close();

        return vResVal;
    }

    /**
     * Copies all columns from one record into a hashtable.
     * The column names are used as keys in the hashtable.
     *                      
     * @param   rsRecord_   The result set pointing via next to the current record.
     * @param   htRecord_   The hashtable used to copy over the record.
     *
     * @exception   SQLException   if anything goes wrong.
     */
    static public void copyRecord( ResultSet rsRecord_
                                   , Hashtable htRecord_ )
        throws SQLException
    {
        ResultSetMetaData metaData = rsRecord_.getMetaData();
        int numberOfColumns = metaData.getColumnCount();
        for( int column = 1; column <= numberOfColumns; column++ ) 
{
            String sColumnName = metaData.getColumnName( column );
            String sValue = rsRecord_.getString( sColumnName );
            htRecord_.put( sColumnName, sValue );
        }
    }

    /**
     * Gets the next id for a given table.
     * The table needs a column with name "id".
     *
     * @exception   SQLException   if anything goes wrong.
     */
    static public String getNextId( Statement statement, String sTableName ) 
        throws SQLException
    {
        String sNextId = "1";
        ResultSet result = statement.executeQuery( "select max(id) from " + sTableName );
        if ( result.next() ) 
{
            sNextId = Util.itoa( result.getInt( 1 ) + 1 );
        }
        result.close();

        return sNextId;
    }
}
