// (c) Microsoft Corporation 2005-2007. 

#light

namespace Microsoft.FSharp.Tools.FsYacc

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Tools.FsLex
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Compatibility
open System

type IParseState<'pos> = 
    interface
        abstract StartOfRHS: int -> 'pos
        abstract EndOfRHS  : int -> 'pos
        abstract StartOfLHS: 'pos
        abstract EndOfLHS  : 'pos
        abstract GetData   : int -> obj 
        abstract RaiseError<'b> : unit -> 'b 
    end

//-------------------------------------------------------------------------
// This is the data structure emitted as code by FSYACC.  

type Tables<'tok,'pos> = 
    { reductions: (IParseState<'pos> -> obj) array;
      end_of_input_tag: int;
      tagof: 'tok -> int;
      dataof: 'tok -> obj; 
      action_table_elements: byte[];  
      action_table_row_offsets: byte[];
      reduction_nsyms: byte[];
      immediate_action: byte[];
      gotos: byte[];
      gotos_row_offsets: byte[];
      nonterms: byte[];
      parse_error:  string -> unit;
      tagof_error_term: int }

//-------------------------------------------------------------------------
// An implementation of stacks.

module Stack = 
    type 'a stack = { mutable contents: 'a array; mutable count: int }

    let ensure (buf: 'a stack) new_size = 
        let old_buf_size = Array.length buf.contents
        if new_size > old_buf_size then 
            let old = buf.contents
            buf.contents <- Array.zero_create (max new_size (old_buf_size * 2));
            Array.blit old 0 buf.contents 0 buf.count;
    
    let pop s = s.count <- s.count - 1
    let peep s = Array.get s.contents (s.count - 1)
    let push s x =
        ensure s (s.count + 1); 
        Array.set s.contents s.count x; 
        s.count <- s.count + 1
    let is_empty s = (s.count = 0)
    let create n = { contents = Array.zero_create n; count = 0 }
    let printStack s = for i = 0 to (s.count - 1) do System.Console.Write("{0}{1}",(Array.get s.contents i),if i=s.count-1 then ":" else "-") done

exception RecoverableParseError
exception Accept of obj

module Implementation = 

    let debug = false
    open Microsoft.FSharp.Text.StructuredFormat.Display
    
    // Definitions shared with fsyacc 
    let anyMarker = 0xffff
    let shiftFlag = 0x0000
    let reduceFlag = 0x4000
    let errorFlag = 0x8000
    let acceptFlag = 0xc000
    let actionMask = 0xc000

    //-------------------------------------------------------------------------
    // Read the tables written by FSYACC.  

    let read_coded_u16  (bytes:byte[]) n =                                    
        let elemSize = 2                                           
        let v0 = int (Bytearray.get bytes (n*elemSize))    
        let v1 = int (Bytearray.get bytes ((n*elemSize)+1))
        v0 * 256 + v1                                                  

    let read_coded_u16_matrix  bytes n1 n2dim n2 =                   
        read_coded_u16 bytes (n1 * n2dim + n2)                         

    let read_coded_u16_array bytes n =                               
        read_coded_u16 bytes n                                         

    let rec coded_u16_u16_assoc elemTab minElemNumber maxElemNumber defaultValueOfAssoc keyToFind =     
        // do a binary chop on the table 
        let elemNumber : int = (minElemNumber+maxElemNumber)/2
        //if debug then System.Console.WriteLine("elemNumber = {0}",elemNumber);
        if elemNumber = maxElemNumber 
        then defaultValueOfAssoc
        else 
            let x = read_coded_u16 elemTab (elemNumber*2)              
            if   keyToFind = x then 
                //if debug then System.Console.WriteLine("found!");
                read_coded_u16 elemTab (elemNumber*2+1)                      
            elif keyToFind < x then coded_u16_u16_assoc elemTab minElemNumber  elemNumber    defaultValueOfAssoc keyToFind       
            else                    coded_u16_u16_assoc elemTab (elemNumber+1) maxElemNumber defaultValueOfAssoc keyToFind       

    let read_coded_u16_u16_assoc_table elemTab offsetTab n keyToFind =       
        let headOfTable = read_coded_u16 offsetTab n           
        let firstElemNumber = headOfTable + 1           
        let numberOfElementsInAssoc = read_coded_u16 elemTab (headOfTable*2)           
        let defaultValueOfAssoc = read_coded_u16 elemTab (headOfTable*2+1)           
        coded_u16_u16_assoc elemTab firstElemNumber (firstElemNumber+numberOfElementsInAssoc) defaultValueOfAssoc keyToFind                 

    //-------------------------------------------------------------------------
    // interpret the tables emitted by FSYACC.  

    let interpret 
          (tables: Tables<'tok,'pos>)
          lexer 
          (lexbuf : LexBuffer<_,_>)
          initialState =
        if debug then System.Console.WriteLine("\nParser: interpret tables");
        let stateStack : int Stack.stack = Stack.create 100
        Stack.push stateStack initialState;
        let valueStack = Stack.create 100                           
        let startPosStack = Stack.create 100                           
        let endPosStack = Stack.create 100                           
        let mutable haveLookahead = false                                                                              
        let mutable lookaheadToken = Unchecked.defaultof<'tok>
        let mutable lookaheadEndPos = Unchecked.defaultof<'pos>
        let mutable lookaheadStartPos = Unchecked.defaultof<'pos>
        let mutable finished = false
        let mutable errorRecoveryPointLevel = 0
        // The 100 here means a maximum of 100 elements for each rule
        let ruleStartPoss = (Array.zero_create 100 : 'pos array)              
        let ruleEndPoss = (Array.zero_create 100 : 'pos array)              
        let ruleValues = (Array.zero_create 100 : obj array)              
        let lhsPos = (Array.zero_create 2 : 'pos array)                                            
        let reductions = tables.reductions                                                                
        let provider =                                                                                            
            { new IParseState<_> with 
                member p.StartOfRHS(n) = ruleStartPoss.(n-1); 
                member p.GetData(n) = ruleValues.(n-1);        
                member p.EndOfRHS(n) = ruleEndPoss.(n-1);     
                member p.StartOfLHS = lhsPos.(0);   
                member p.EndOfLHS   = lhsPos.(1);
                member p.RaiseError()= raise RecoverableParseError  (* NOTE: this binding tests the fairly complex logic associated with an object expression implementing a generic abstract method *)
            }

        let report haveLookahead lookaheadToken = if haveLookahead then any_to_string lookaheadToken else "[TBC]"

        let popStackUntilErrorShifted() =
            if debug then System.Console.WriteLine("popStackUntilErrorShifted");
            let mutable errorShifted = false
            while not errorShifted do
                if Stack.is_empty valueStack then 
                    if debug then 
                        System.Console.WriteLine("stack empty during error recovery - generating parse error");
                    failwith "parse error";
                
                let stateNow = Stack.peep stateStack
                if debug then 
                    System.Console.WriteLine("In state {0} during error recovery", stateNow);
                
                let action = 
                    read_coded_u16_u16_assoc_table                                                         
                        tables.action_table_elements                                                   
                        tables.action_table_row_offsets stateNow tables.tagof_error_term                                      
                
                let kind = action &&& actionMask                                             
                if kind = shiftFlag then
                    if debug then System.Console.WriteLine("shifting error, continuing with error recovery");
                    errorShifted <- true
                    let dest = action &&& (~~~ actionMask)                                    
                    Stack.push valueStack (box ())
                    Stack.push startPosStack (Stack.peep startPosStack)
                    Stack.push endPosStack (Stack.peep endPosStack)
                    Stack.push stateStack dest
                else
                    if debug then 
                        System.Console.WriteLine("popping stack during error recovery");
                    Stack.pop valueStack;
                    Stack.pop startPosStack;                                                                          
                    Stack.pop endPosStack;                                                                           
                    Stack.pop stateStack;
            done

        while not finished do                                                                                    
            if Stack.is_empty stateStack then 
                finished <- true
            else
                let s = Stack.peep stateStack
                if debug then (Console.Write("{0} value(s), state ",valueStack.count); Stack.printStack stateStack)
                let action = 
                    let immediateAction = read_coded_u16 tables.immediate_action s // check: s correct? 
                    if not (immediateAction = anyMarker) then
                        // Action has been pre-determined, no need to lookahead 
                        // Expecting it to be a Reduce action on a non-fakeStartNonTerminal ? 
                        immediateAction
                    else
                        // Lookahead required to determine action 
                          if not haveLookahead && not lexbuf.IsPastEndOfStream then 
                              lookaheadToken <- lexer lexbuf;
                              haveLookahead <- true;
                              lookaheadStartPos <- lexbuf.StartPos;
                              lookaheadEndPos <- lexbuf.EndPos;
                          let tag = if haveLookahead then tables.tagof lookaheadToken else tables.end_of_input_tag   
                          // Printf.printf "state %d\n" s  
                          let action = 
                              read_coded_u16_u16_assoc_table                                                         
                                  tables.action_table_elements                                                   
                                  tables.action_table_row_offsets s tag
                          action
                let kind = action &&& actionMask                                             
                if kind = shiftFlag then 
                    if errorRecoveryPointLevel > 0 then 
                        errorRecoveryPointLevel <- errorRecoveryPointLevel - 1;
                        if debug then Console.WriteLine("shifting, reduced errorRecoveryPointlevel to {0}\n", errorRecoveryPointLevel);
                    let dest = action &&& (~~~ actionMask)                                    
                    if not haveLookahead then failwith "shift on end of input!";
                    Stack.push valueStack (tables.dataof lookaheadToken);
                    Stack.push startPosStack lookaheadStartPos;
                    Stack.push endPosStack lookaheadEndPos;
                    Stack.push stateStack dest;
                    if debug then Console.WriteLine("shift/consume input {0}, shift to state {1}", report haveLookahead lookaheadToken, dest);
                    haveLookahead <- false
                elif kind = reduceFlag then
                    let prod = action &&& (~~~ actionMask)                                    
                    let reduction = reductions.(prod)                                                             
                    let n = read_coded_u16_array  tables.reduction_nsyms prod                                    
                       // pop the symbols, populate the values and populate the locations
                    if debug then Console.Write("reduce popping {0} values/states, lookahead {1}", n, report haveLookahead lookaheadToken);
                    for i = 0 to n - 1 do
                        if Stack.is_empty valueStack then failwith "empty symbol stack";
                        let topVal = Stack.peep valueStack
                        let topStartPos = Stack.peep startPosStack
                        let topEndPos = Stack.peep endPosStack
                        Stack.pop valueStack;
                        Stack.pop startPosStack;                                                                           
                        Stack.pop endPosStack;                                                                           
                        Stack.pop stateStack;
                        ruleValues.((n - i) - 1) <- topVal;  
                        ruleStartPoss.((n - i) - 1) <- topStartPos;  
                        ruleEndPoss.((n - i) - 1) <- topEndPos;  
                        if i = 0 then lhsPos.(1) <- topEndPos;                                     
                        if i = n - 1 then lhsPos.(0) <- topStartPos                             
                    done;                                                                                             
                    // Use the lookahead token to populate the locations if the rhs is empty                        
                    if n = 0 then 
                        if haveLookahead then 
                           lhsPos.(0) <- lookaheadStartPos;                                                                     
                           lhsPos.(1) <- lookaheadEndPos;                                                                       
                        else 
                           lhsPos.(0) <- lexbuf.StartPos;
                           lhsPos.(1) <- lexbuf.EndPos;
                    try                                                                                               
                          // Printf.printf "reduce %d\n" prod;                                                       
                        let redResult = reduction provider                                                          
                        Stack.push valueStack redResult;
                        Stack.push startPosStack lhsPos.(0);
                        Stack.push endPosStack lhsPos.(1);
                        let stateNow = Stack.peep stateStack
                        let newGotoState =                                                                              
                          read_coded_u16_u16_assoc_table                                                             
                            tables.gotos                                                                            
                            tables.gotos_row_offsets                                                               
                            (read_coded_u16 tables.nonterms prod) stateNow                                      
                        Stack.push stateStack newGotoState
                        if debug then Console.WriteLine(" goto state {0}", newGotoState)
                    with                                                                                              
                    | Accept res ->                                                                            
                          finished <- true;                                                                             
                          Stack.push valueStack res;
                          Stack.push startPosStack lhsPos.(0);
                          Stack.push endPosStack lhsPos.(1) 
                    | RecoverableParseError ->
                          if debug then Console.WriteLine("RecoverableParseError...\n");
                          popStackUntilErrorShifted();
                          errorRecoveryPointLevel <- 3
                elif kind = errorFlag then
                    if debug then Console.Write("ErrorFlag... ");
                    if errorRecoveryPointLevel > 0 then 
                        if not haveLookahead then failwith "end of input";
                        if debug then System.Console.WriteLine("unrecognized input {0}, state = {1}, discarding, reset errorRecoveryPointlevel from {2} to 3\n", report haveLookahead lookaheadToken, s, errorRecoveryPointLevel);
                        haveLookahead <- false;
                        errorRecoveryPointLevel <- 3
                    else
                        tables.parse_error "syntax error";
                        popStackUntilErrorShifted();
                        if debug then System.Console.WriteLine("generated syntax error and shifted error token, haveLookahead = {0}\n", haveLookahead);
                        errorRecoveryPointLevel <- 3
                elif kind = acceptFlag then 
                    finished <- true
                else
                  if debug then System.Console.WriteLine("ALARM!!! drop through case in parser");  
        done;                                                                                                     
        Stack.peep valueStack

type Tables<'tok,'pos> 
    with
        member tables.Interpret (lexer,lexbuf,initialState) = Implementation.interpret tables lexer lexbuf initialState
    end
    
module ParseHelpers = 
    let parse_error (s:string) = ()

