// (c) Microsoft Corporation 2005-2007. 

#light

/// Basic F# type definitions and F#-related libraries
namespace Microsoft.FSharp.Core

    open System

    type obj = System.Object

    type exn = System.Exception

    type nativeint = System.IntPtr
    type unativeint = System.UIntPtr

    type string = System.String

    type float32 = System.Single
    type float = System.Double
    type single = System.Single
    type double = System.Double

    type sbyte = System.SByte
    type byte = System.Byte
    type int8 = System.SByte
    type uint8 = System.Byte

    type int16 = System.Int16
    type uint16 = System.UInt16

    type int32 = System.Int32
    type uint32 = System.UInt32

    type int64 = System.Int64
    type uint64 = System.UInt64

    type char = System.Char
    type bool = System.Boolean
    type decimal = System.Decimal

    type int = int32

    /// Single dimensional, zero-based arrays, written 'int[]', 'string[]' etc.
    /// Use the values in the [[Array]] module to manipulate values 
    /// of this type, or the notation 'arr.[x]' to get/set array
    /// values.   When using .NET 1.x use the values in the [[CompatArray]]
    /// module.
    type 'a ``[]`` = (# "!0[]" #)

    /// Two dimensional arrays, typically zero-based. 
    ///
    /// Use the values in the [[Array2]] module
    /// to manipulate values of this type, or the notation 'arr.[x,y]' to get/set array
    /// values.   When using .NET 1.x use the values in the [[CompatArray2]]
    /// module.
    ///
    /// Non-zero-based arrays can be created using methods on the System.Array type.
    type 'a ``[,]`` = (# "!0[,]" #)

    /// Three dimensional arrays, typically zero-based. Non-zero-based arrays
    /// can be created using methods on the System.Array type.
    type 'a ``[,,]`` = (# "!0[,,]" #)

    /// Four dimensional arrays, typically zero-based. Non-zero-based arrays
    /// can be created using methods on the System.Array type.
    type 'a ``[,,,]`` = (# "!0[,,,]" #)

    #if CLI_AT_LEAST_2_0
    /// Single dimensional, zero-based arrays, written 'int[]', 'string[]' etc.
    /// Use the values in the [[Array]] module to manipulate values 
    /// of this type, or the notation 'arr.[x]' to get/set array
    /// values.   When using .NET 1.x use the values in the [[CompatArray]]
    /// module.
    type 'a array = 'a[]
    #else
    /// Single dimensional, zero-based arrays, written 'int[]', 'string[]' etc.
    /// Use the values in the [[Array]] module to manipulate values 
    /// of this type, or the notation 'arr.[x]' to get/set array
    /// values.   When using .NET 1.x use the values in the [[CompatArray]]
    /// module.
    type 'a array = (# "!0[]!" #) // ILX erasable array type
    #endif
            
           
    //-------------------------------------------------------------------------
    // By-reference stack-allocated argument passing, used by IStructuralHash to
    // pass the hash-node-count around.

    /// The type for by-reference, stack-allocated argument passing.  It is used
    /// by .NET assemblies and by the type [[IStructuralHash]] to
    /// pass the hash-node-count around.  Use of this type in F# code may result in
    /// unverifiable code being generated.
    type 'a byref = (# "!0&" #)

    /// The type of machine pointers assumed to be pointing to 
    /// unmanaged or pinned memory containing values or an array of
    /// values of the given type.
    ///
    /// This type should only be used when writing F# code that interoperates
    /// with C code.  Use of this type in F# code may result in
    /// unverifiable code being generated.  Conversions to and from the 
    /// [[nativeint]] type may be required. Values of this type can be generated
    /// by the functions in the NativeInterop.NativePtr module.
    type 'a nativeptr = (# "native int" #)

    /// This type is for internal use by the F# code generator
    type 'a ilsigptr = (# "!0*" #)

    type IStructuralHash =
        abstract GetStructuralHashCode : nodesRemaining: int byref -> int

    /// The type 'unit', which has only one value "()". This value is special and
    /// always uses the representation 'null'.
    and Unit with 
            interface System.IComparable 
            interface IStructuralHash 
        
    /// The type 'unit', which has only one value "()". This value is special and
    /// always uses the representation 'null'.
    and unit = Unit

    type SourceLevelConstruct = 
       | Unspecified = 0
       | SumType = 1
       | RecordType = 2
       | ObjectType = 3
       | Field = 4
       | Exception = 5
       | Closure = 6
       | Module = 7
       | Alternative = 8
       | Value = 9

    type CompilationRepresentationFlags = 
       /// No special compilation representation
       | Unspecified = 0
       /// Compile an instance member as 'static' 
       | Static = 1
       /// Compile a member as 'instance' even if 'null' is used as a representation for this type
       | Instance = 2
       /// append 'Module' to the end of a module whose name clashes with a type name in the same namespace
       | ModuleSuffix = 4  
       /// Permit the use of 'null' as a representation for nullary discriminators in a discriminated union
       | PermitNull = 8

    //-------------------------------------------------------------------------
    // F#-specific Attributes


    /// Adding this attribute to class definition makes it sealed, which means it may not
    /// be extended or implemented.
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type SealedAttribute =
        inherit System.Attribute
        new : unit -> SealedAttribute

    /// Adding this attribute to class definition makes it abstract, which means it need not
    /// implement all its methods. Instances of abstract classes may not be constructed directly.
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type AbstractClassAttribute =
        inherit System.Attribute
        new : unit -> AbstractClassAttribute


    /// Adding this attribute to the let-binding for the definition of a top-level 
    /// value makes the quotation expression that implements the value available
    /// for use at runtime.
    [<System.AttributeUsage (System.AttributeTargets.Method,AllowMultiple=false)>]  
    type ReflectedDefinitionAttribute =
        inherit System.Attribute
        new : unit -> ReflectedDefinitionAttribute

    /// Adding this attribute to a type causes it to be represented using a .NET struct.
    [<System.AttributeUsage (System.AttributeTargets.Struct,AllowMultiple=false)>]  
    type StructAttribute =
        inherit System.Attribute
        new : unit -> StructAttribute

    /// Adding this attribute to a type causes it to be represented using a .NET interface.
    [<System.AttributeUsage (System.AttributeTargets.Interface,AllowMultiple=false)>]  
    type InterfaceAttribute =
        inherit System.Attribute
        new : unit -> InterfaceAttribute

    /// Adding this attribute to a type causes it to be represented using a .NET class.
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type ClassAttribute =
        inherit System.Attribute
        new : unit -> ClassAttribute

    /// Adding this attribute to a value causes it to be compiled as a .NET constant literal.
    [<System.AttributeUsage (System.AttributeTargets.Field,AllowMultiple=false)>]  
    type LiteralAttribute =
        inherit System.Attribute
        new : unit -> LiteralAttribute

    /// Adding this attribute to a discriminated union with value false
    /// turns off the generation of standard helper member tester, constructor 
    /// and accessor members for the generated .NET class for that type.
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type DefaultAugmentationAttribute =
        inherit System.Attribute
        member Value: bool
        new : bool -> DefaultAugmentationAttribute

    /// Adding this attribute to a function indicates it is the entrypoint for an application.
    /// If this absent is not speficied for an EXE then the initialization implicit in the
    /// module bindings in the last file in the compilation sequence are used as the entrypoint.
    [<System.AttributeUsage (System.AttributeTargets.Method,AllowMultiple=false)>]  
    type EntryPointAttribute =
        inherit System.Attribute
        new : unit -> EntryPointAttribute

    /// Adding this attribute to a record or union type disables the automatic generation
    /// of overrides for 'System.Object.Equals(obj)', 'System.Object.GetHashCode()' 
    /// and 'System.IComparable' for the type. The type will by default use reference equality.
    /// This is identical to adding attributes StructuralEquality(false) and StructuralComparison(false).
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type ReferenceEqualityAttribute =
        inherit System.Attribute
        member Value: bool
        new : unit -> ReferenceEqualityAttribute
        new : value:bool -> ReferenceEqualityAttribute

    /// Adding this attribute to a record, union or struct type with value 'false' 
    /// confirms the automatic generation of overrides for 'System.Object.Equals(obj)' 
    /// and 'System.Object.GetHashCode()' for the type. This attribute is usually used in 
    /// conjunction with StructuralComparison(false) to generate a type that supports
    /// structural equality but not structural comparison.
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type StructuralEqualityAttribute =
        inherit System.Attribute
        member Value: bool
        new : unit -> StructuralEqualityAttribute
        new : value:bool -> StructuralEqualityAttribute

    /// Adding this attribute to a record, union or struct type with value 'false' disables the automatic generation
    /// of implementations for 'System.IComparable' for the type.
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type StructuralComparisonAttribute =
        inherit System.Attribute
        member Value: bool
        new : unit -> StructuralComparisonAttribute
        new : value:bool -> StructuralComparisonAttribute

    /// Adding this attribute to a field declaration 
    /// means that the field is not initialized in the turns off the generation of standard helper member tester, constructor 
    /// and accessor members for the generated .NET class for that type. During 
    /// type checking constraint is asserted that the field type support 'null'. 
    /// If the 'check' value is false then the constraint is not asserted. 
    [<System.AttributeUsage (System.AttributeTargets.Field,AllowMultiple=false)>]  
    type DefaultValueAttribute =
        inherit System.Attribute
        member Check: bool
        new : unit -> DefaultValueAttribute
        new : check: bool -> DefaultValueAttribute

    /// This attribute is added automatically for all optional arguments
    [<System.AttributeUsage (System.AttributeTargets.Parameter,AllowMultiple=false)>]  
    type OptionalArgumentAttribute =
        inherit System.Attribute
        new : unit -> OptionalArgumentAttribute

    /// Adding this attribute to a type, value or member requires that 
    /// uses of the construct must explicitly instantiate any generic type parameters.
    [<System.AttributeUsage (System.AttributeTargets.Method,AllowMultiple=false)>]  
    type RequiresExplicitTypeArgumentsAttribute =
        inherit System.Attribute
        new : unit -> RequiresExplicitTypeArgumentsAttribute

    /// Adding this attribute to a non-function value with generic parameters indicates that 
    /// uses of the construct can give rise to generic code through type inference. 
    [<System.AttributeUsage (System.AttributeTargets.Method,AllowMultiple=false)>]  
    type GeneralizableValueAttribute =
        inherit System.Attribute
        new : unit -> GeneralizableValueAttribute

    /// Adding the OverloadID attribute to a member permits it to
    /// be part of a group overloaded by the same name and arity.  The string
    /// must be a unique name amongst those in the overload set.  Overrides
    /// of this method, if permitted, must be given the same OverloadID,
    /// and the OverloadID must be specified in both signature and implementation
    /// files if signature files are used.
    [<System.AttributeUsage (System.AttributeTargets.Method,AllowMultiple=false)>]  
    type OverloadIDAttribute =
        inherit System.Attribute
        /// A unique identifier for this overloaded member within a given overload set
        member UniqueName: string
        new : uniqueName:string -> OverloadIDAttribute

    /// Adding this attribute to a type with value 'false' disables the behaviour where F# makes the
    /// type Serializable by default.
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type AutoSerializableAttribute =
        inherit System.Attribute
        member Value: bool
        new : value:bool -> AutoSerializableAttribute

    /// This attribute is added to generated assemblies to indicate the 
    /// version of the data schema used to encode additional F#
    /// specific information in the resource attached to compiled F# libraries.
    [<System.AttributeUsage (System.AttributeTargets.Assembly,AllowMultiple=false)>]  
    type FSharpInterfaceDataVersionAttribute =
        inherit System.Attribute
        member Major: int
        member Minor: int
        member Release: int
        new : major:int * minor:int * release:int -> FSharpInterfaceDataVersionAttribute

    /// This attribute is inserted automatically by the F# compiler to tag 
    /// types and methods in the gneerated .NET code with flags indicating the correspondence with
    /// original source constructs.  It is used by the functions in the
    /// Microsoft.FSharp.Reflection library to reverse-map compiled constructs
    /// to their original forms.  It is not intended for use from use code.
    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type CompilationMappingAttribute =
        inherit System.Attribute
        member SourceConstruct : SourceLevelConstruct
        member SequenceNumber : int
        member VariantNumber : int
        new : sourceConstruct:SourceLevelConstruct -> CompilationMappingAttribute
        new : sourceConstruct:SourceLevelConstruct * sequenceNumber: int -> CompilationMappingAttribute
        new : sourceConstruct:SourceLevelConstruct * variantNumber : int * sequenceNumber : int -> CompilationMappingAttribute

    /// This attribute is used to adjust the runtime representation for a type. 
    /// For example, it may be used to note that the 'null' representation
    /// may be used for a type.  This affects how some constructs are compiled.
    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type CompilationRepresentationAttribute =
        inherit System.Attribute
        member Flags : CompilationRepresentationFlags
        new : flags:CompilationRepresentationFlags -> CompilationRepresentationAttribute

    /// This attribute is used to tag values that are part of an experimental library
    /// feature
    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type ExperimentalAttribute =
        inherit System.Attribute
        member Message: string
        new : message:string-> ExperimentalAttribute

    /// This attribute is used to tag values, modules and types that are only
    /// present in F# to permit a degree of code-compatibility and cross-compilation
    /// with other implementations of ML-familty languages, in particular OCaml. The
    /// use of the construct will give a warning unless the --ml-compatibility flag
    /// is specified.
    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type OCamlCompatibilityAttribute =
        inherit System.Attribute
        member Message: string
        new : unit -> OCamlCompatibilityAttribute
        new : message:string -> OCamlCompatibilityAttribute

    /// This attribute is used to tag values whose use will result in the generation
    /// of unverifiable code. These values are inevitably marked 'inline' to ensure that
    /// the unverifiable constructs are not present in the actual code for the F# library,
    /// but are rather copied to the source code of the caller.
    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type UnverifiableAttribute =
        inherit System.Attribute
        new : unit -> UnverifiableAttribute

    //-------------------------------------------------------------------------
    // Compiler-emitted language primitives


    module LanguagePrimitives : 


        /// Compare two values for equality
        val inline GenericEquality : 'a -> 'a -> bool

        /// Compare two values 
        val inline GenericComparison : 'a -> 'a -> int

        /// Compare two values for disequality
        [<Obsolete("Use 'not(GenericEquality x y)' instead")>]
        val inline GenericInequality : 'a -> 'a -> bool

        /// Compare two values   
        val inline GenericLessThan : 'a -> 'a -> bool

        /// Compare two values   
        val inline GenericGreaterThan : 'a -> 'a -> bool

        /// Compare two values   
        val inline GenericLessOrEqual : 'a -> 'a -> bool

        /// Compare two values   
        val inline GenericGreaterOrEqual : 'a -> 'a -> bool

        /// Take the minimum of two values structurally according to the order given by GenericComparison
        val inline GenericMinimum : 'a -> 'a -> 'a

        /// Take the maximum of two values structurally according to the order given by GenericComparison
        val inline GenericMaximum : 'a -> 'a -> 'a


        /// Reference/physical equality. 
        /// True if boxed versions of the inputs are reference-equal, OR if
        /// both are primitive numeric types and the implementation of Object.Equals for the type
        /// of the first argument returns true on the boxed versions of the inputs.  
        val inline PhysicalEquality : 'a -> 'a -> bool

        /// Reference/physical disequality (not equals)
        /// False if boxed versions of the inputs are reference-equal, OR if
        /// both are primitive numeric types and the implementation of Object.Equals for the type
        /// of the first argument returns true on the boxed versions of the inputs.  
        [<Obsolete("Use 'not(PhysicalEquality x y)' instead")>]
        val inline PhysicalInequality : 'a -> 'a -> bool

        /// The physical hash.  Hashes on the object identity, except for value types,
        /// where we hash on the contents.
        val inline PhysicalHash : 'a -> int

        /// Hash a value according to its structure.  See the IStructuralHash interface.
        /// Implementations of IStructuralHash for aggregate structured data structures
        /// should call this function to hash sub-structures.
        ///
        /// The byref argument points to a count of the
        /// number of significant nodes remaining to be hashed in the cooperative hash.
        /// Substructures and leaf nodes (such as integers) should be hashed by calling 
        /// Microsoft.FSharp.Core.LanguagePrimitives.GenericHashParam, but only if the hash count
        /// is non-zero.  If the hash count is zero GenericHashParam must not be called.
        [<Obsolete("Use GenericHash instead")>]
        val inline GenericHashParam : 'a -> int byref -> int
        
        val inline GenericHash : 'a -> int

        /// Generate a null value for reference types.
        val inline DefaultValue<'a when 'a : null> : 'a

        [<Obsolete("This function has been renamed to GenericEquality")>]
        val inline StructuralEquality : 'a -> 'a -> bool

        [<Obsolete("This function has been renamed to GenericComparison")>]
        val inline StructuralComparison : 'a -> 'a -> int

        [<Obsolete("Use 'not(GenericEquality x y)' instead")>]
        val inline StructuralInequality : 'a -> 'a -> bool

        [<Obsolete("This function has been renamed to GenericLessThan")>]
        val inline StructuralLessThan : 'a -> 'a -> bool

        [<Obsolete("This function has been renamed to GenericGreaterThan")>]
        val inline StructuralGreaterThan : 'a -> 'a -> bool

        [<Obsolete("This function has been renamed to GenericLessOrEqual")>]
        val inline StructuralLessOrEqual : 'a -> 'a -> bool

        [<Obsolete("This function has been renamed to GenericGreaterOrEqual")>]
        val inline StructuralGreaterOrEqual : 'a -> 'a -> bool

        [<Obsolete("This function has been renamed to GenericMinimum")>]
        val inline StructuralMinimum : 'a -> 'a -> 'a

        [<Obsolete("This function has been renamed to GenericMaximum")>]
        val inline StructuralMaximum : 'a -> 'a -> 'a

        [<Obsolete("This function has been renamed to GenericHashParam")>]
        val inline StructuralHashParam : 'a -> int byref -> int
        
        [<Obsolete("This function has been renamed to GenericHash")>]
        val inline StructuralHash : 'a -> int

        /// Build an enum value from an underlying value. 
        val inline EnumOfValue : 'u -> 'e when 'e : enum<'u>
        val inline EnumToValue : 'e -> 'u when 'e : enum<'u>

        //-------------------------------------------------------------------------

        /// Compiler-recognized operators
        module IntrinsicOperators = 

            /// Binary 'and'.  When used as a binary operator the right hand value is evaluated only on demand
            [<OCamlCompatibility("Use '&&' instead")>]
            val ( & ) : bool -> bool -> bool
            /// Binary 'and'.  When used as a binary operator the right hand value is evaluated only on demand
            val ( && ) : bool -> bool -> bool
            /// Binary 'or'.  When used as a binary operator the right hand value is evaluated only on demand
            //[<OCamlCompatibility("Use '||' instead")>]
            val ( or ) : bool -> bool -> bool
            /// Binary 'or'.  When used as a binary operator the right hand value is evaluated only on demand
            val ( || ) : bool -> bool -> bool
            /// Address-of. Uses of this value may result in the generation of unverifiable code.
            val inline ( ~& ) : 'a -> 'a byref
            /// Address-of. Uses of this value may result in the generation of unverifiable code.
            val inline ( ~&& ) : 'a -> 'a nativeptr
            /// Overloaded unary negation.
            val inline ( ~- ) : ^a -> ^a when ^a : (static member ( ~- ) : ^a -> ^a) and default ^a : int

        //-------------------------------------------------------------------------

        // Compiler-generated primitives
        module IntrinsicFunctions = 

#if CLI_AT_MOST_1_1
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline UnboxGeneric< 'a > : obj -> 'a
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline UnboxFast< 'a > : obj -> 'a
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline TypeTestGeneric< $a > : obj -> bool
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline TypeTestFast< $a > : obj -> bool 
#else

            /// A compiler intrinsic that implements the ':?>' operator
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val UnboxGeneric<'a> : obj -> 'a

            /// A compiler intrinsic that implements the ':?>' operator
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline UnboxFast<'a> : obj -> 'a

            /// A compiler intrinsic that implements the ':?' operator
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val TypeTestGeneric<'a> : obj -> bool

            /// A compiler intrinsic that implements the ':?' operator
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline TypeTestFast<'a> : obj -> bool 
#endif

            /// Primitive used by pattern match compilation 
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline ArrayGet : 'a array -> int -> 'a

            /// This function implements calls to default constructors
            /// acccessed by 'new' constraints.
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
#if CLI_AT_MOST_1_1
            val inline CreateInstance : unit -> $a when $a : (new : unit -> $a)
#else
            val inline CreateInstance : unit -> 'a when 'a : (new : unit -> 'a)
#endif
            /// This function implements parsing of decimal constants
            [<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline ParseDecimal : string -> decimal

            /// The standard overloaded associative (indexed) lookup operator
            //[<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline (.[])     : ^src -> 'idx -> 'elem                           when ^src : (member get_Item   : 'idx                  -> 'elem) 
            /// The standard overloaded associative (2-indexed) lookup operator
            //[<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline (.[,])    : ^src -> 'idx1 * 'idx2 -> 'elem                  when ^src : (member get_Item   : 'idx1 * 'idx2         -> 'elem)
            /// The standard overloaded associative (3-indexed) lookup operator
            //[<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline (.[,,])   : ^src -> 'idx1 * 'idx2 * 'idx3 -> 'elem          when ^src : (member get_Item   : 'idx1 * 'idx2 * 'idx3 -> 'elem)
            /// The standard overloaded associative (indexed) mutation operator
            //[<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline (.[]<-)   : ^src -> 'idx              -> 'elem -> unit      when ^src : (member set_Item   : 'idx                  * 'elem -> unit) 
            /// The standard overloaded associative (2-indexed) mutation operator
            //[<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline (.[,]<-)  : ^src -> 'idx1 * 'idx2       -> 'elem -> unit    when ^src : (member set_Item   : 'idx1 * 'idx2         * 'elem -> unit)
            /// The standard overloaded associative (3-indexed) mutation operator
            //[<Obsolete("This function is for use by compiled F# code and should not be used directly")>]
            val inline (.[,,]<-) : ^src -> 'idx1 * 'idx2 * 'idx3 -> 'elem -> unit  when ^src : (member set_Item   : 'idx1 * 'idx2 * 'idx3 * 'elem -> unit)

            //-------------------------------------------------------------------------
            // Compiler-recognized equality primitives

        
        module HashCompare : 
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val PhysicalHashIntrinsic : 'a -> int
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val PhysicalEqualityIntrinsic : 'a -> 'a -> bool
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val GenericHashParamIntrinsic : 'a -> int byref -> int
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val GenericComparisonIntrinsic : 'a -> 'a -> int
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val GenericEqualityIntrinsic : 'a -> 'a -> bool
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val GenericLessThanIntrinsic : 'a -> 'a -> bool
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val GenericGreaterThanIntrinsic : 'a -> 'a -> bool
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val GenericGreaterOrEqualIntrinsic : 'a -> 'a -> bool
            /// A primitive entry point used by the F# compiler for optimization purposes. 
            [<Obsolete("This function is a primitive library routine used by optimized F# code and should not be used directly")>]
            val GenericLessOrEqualIntrinsic : 'a -> 'a -> bool



    //-------------------------------------------------------------------------
    // F# Tuple Types


    /// Compiled versions of F# tuple types.  These are not used directly, though
    /// these compiled forms are seen by other .NET languages.

    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2})")>]
    #endif
    type Tuple<'a,'b>                = { Item1: 'a; Item2: 'b }

    /// Compiled versions of F# tuple types.  These are not used directly, though
    /// these compiled forms are seen by other .NET languages.
    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3})")>]
    #endif
    type Tuple<'a,'b,'c>             = { Item1: 'a; Item2: 'b; Item3: 'c }

    /// Compiled versions of F# tuple types.  These are not used directly, though
    /// these compiled forms are seen by other .NET languages.
    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4})")>]
    #endif
    type Tuple<'a,'b,'c,'d>          = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd }

    /// Compiled versions of F# tuple types.  These are not used directly, though
    /// these compiled forms are seen by other .NET languages.
    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4},{Item5})")>]
    #endif
    type Tuple<'a,'b,'c,'d,'e>       = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd ; Item5: 'e  }

    /// Compiled versions of F# tuple types.  These are not used directly, though
    /// these compiled forms are seen by other .NET languages.
    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4},{Item5},{Item6})")>]
    #endif
    type Tuple<'a,'b,'c,'d,'e,'f>    = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd ; Item5: 'e ; Item6: 'f   }

    /// Compiled versions of F# tuple types.  These are not used directly, though
    /// these compiled forms are seen by other .NET languages.
    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4},{Item5},{Item6},{Item7})")>]
    #endif
    type Tuple<'a,'b,'c,'d,'e,'f,'g> = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd ; Item5: 'e ; Item6: 'f ; Item7: 'g  }

    //-------------------------------------------------------------------------
    // F# Choice Types


    /// Helper types for active patterns with 2 choices.
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b> = 
      /// Choice 1 of 2 choices
      | Choice2_1 of 'a 
      /// Choice 2 of 2 choices
      | Choice2_2 of 'b
    
    /// Helper types for active patterns with 3 choices.
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c> = 
      /// Choice 1 of 3 choices
      | Choice3_1 of 'a 
      /// Choice 2 of 3 choices
      | Choice3_2 of 'b
      /// Choice 3 of 3 choices
      | Choice3_3 of 'c
    
    /// Helper types for active patterns with 4 choices.
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d> = 
      /// Choice 1 of 4 choices
      | Choice4_1 of 'a 
      /// Choice 2 of 4 choices
      | Choice4_2 of 'b
      /// Choice 3 of 4 choices
      | Choice4_3 of 'c
      /// Choice 4 of 4 choices
      | Choice4_4 of 'd
    
    /// Helper types for active patterns with 5 choices.
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d,'e> = 
      /// Choice 1 of 5 choices
      | Choice5_1 of 'a 
      /// Choice 2 of 5 choices
      | Choice5_2 of 'b
      /// Choice 3 of 5 choices
      | Choice5_3 of 'c
      /// Choice 4 of 5 choices
      | Choice5_4 of 'd
      /// Choice 5 of 5 choices
      | Choice5_5 of 'e
    
    /// Helper types for active patterns with 6 choices.
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d,'e,'f> = 
      /// Choice 1 of 6 choices
      | Choice6_1 of 'a 
      /// Choice 2 of 6 choices
      | Choice6_2 of 'b
      /// Choice 3 of 6 choices
      | Choice6_3 of 'c
      /// Choice 4 of 6 choices
      | Choice6_4 of 'd
      /// Choice 5 of 6 choices
      | Choice6_5 of 'e
      /// Choice 6 of 6 choices
      | Choice6_6 of 'f
    
    /// Helper types for active patterns with 7 choices.
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d,'e,'f,'g> = 
      /// Choice 1 of 7 choices
      | Choice7_1 of 'a 
      /// Choice 2 of 7 choices
      | Choice7_2 of 'b
      /// Choice 3 of 7 choices
      | Choice7_3 of 'c
      /// Choice 4 of 7 choices
      | Choice7_4 of 'd
      /// Choice 5 of 7 choices
      | Choice7_5 of 'e
      /// Choice 6 of 7 choices
      | Choice7_6 of 'f
      /// Choice 7 of 7 choices
      | Choice7_7 of 'g

    //-------------------------------------------------------------------------
    // F# Exception Types

    
    /// This exception is precisely the same as the exception 'Failure' in F# code.
    exception Failure of string
    /// This exception is precisely the same as the exception 'Invalid_argument' in F# code.
    exception InvalidArgument of string
    /// This exception is raised as the final result of the syntactic expression 'assert(false)'
    exception AssertionFailure of string * int * int
    /// Non-exhaustive match failures will raise the MatchFailure exception
    exception MatchFailure of string * int * int


    //-------------------------------------------------------------------------
    // F# Function Types


    /// The .NET type used to represent F# first-class type function values.  This type is not
    /// typically used directly.
    [<AbstractClass>]
    type TypeFunc =
        abstract Specialize<'T> : unit -> obj
        new : unit -> TypeFunc

    /// The .NET type used to represent F# function values.  This type is not
    /// typically used directly, though may be used from other .NET languages.
    [<AbstractClass>]
    type FastFunc<'T,'U> = 
        abstract member Invoke : 'T -> 'U
#if CLI_AT_LEAST_2_0
        [<OverloadID("ToConverter")>]
        static member op_Implicit : ('T -> 'U) -> System.Converter <'T,'U>
        [<OverloadID("FromConverter")>]
        static member op_Implicit : System.Converter <'T,'U> -> ('T -> 'U)
#endif
        static member InvokeFast5 : FastFunc <'T,('U -> 'c -> 'd -> 'e -> 'f)> * 'T * 'U * 'c * 'd * 'e -> 'f
        static member InvokeFast4 : FastFunc <'T,('U -> 'c -> 'd -> 'e)> * 'T * 'U * 'c * 'd -> 'e
        static member InvokeFast3 : FastFunc <'T,('U -> 'c -> 'd)> * 'T * 'U * 'c -> 'd
        static member InvokeFast2 : FastFunc <'T,('U -> 'c)> * 'T * 'U -> 'c
        new : unit -> FastFunc <'T,'U>

    /// An implementation module used to hold some private implementations of function
    /// value invocation.
    module OptimizedClosures : 

        /// The .NET type used to represent F# function values that accept
        /// two iterated (curried) arguments without intervening execution.  This type should not
        /// typically used directly from either F# code or from other .NET languages.
        [<AbstractClass>]
        type FastFunc2<'T,'U,'V> = 
            inherit FastFunc <'T,('U -> 'V)>
            abstract member Invoke : 'T * 'U -> 'V
            static member Adapt : ('T -> 'U -> 'V) -> FastFunc2<'T,'U,'V>
            new : unit -> FastFunc2 <'T,'U,'V>

        /// The .NET type used to represent F# function values that accept
        /// three iterated (curried) arguments without intervening execution.  This type should not
        /// typically used directly from either F# code or from other .NET languages.
        [<AbstractClass>]
        type FastFunc3<'T,'U,'V,'W> = 
            inherit FastFunc <'T,('U -> 'V -> 'W)>
            abstract member Invoke : 'T * 'U * 'V -> 'W
            static member Adapt : ('T -> 'U -> 'V -> 'W) -> FastFunc3<'T,'U,'V,'W>
            new : unit -> FastFunc3 <'T,'U,'V,'W>

        /// The .NET type used to represent F# function values that accept
        /// four iterated (curried) arguments without intervening execution.  This type should not
        /// typically used directly from either F# code or from other .NET languages.
        [<AbstractClass>]
        type FastFunc4<'T,'U,'V,'W,'X> = 
            inherit FastFunc <'T,('U -> 'V -> 'W -> 'X)>
            abstract member Invoke : 'T * 'U * 'V * 'W -> 'X
            static member Adapt : ('T -> 'U -> 'V -> 'W -> 'X) -> FastFunc4<'T,'U,'V,'W,'X>
            new : unit -> FastFunc4 <'T,'U,'V,'W,'X>

        /// The .NET type used to represent F# function values that accept
        /// five iterated (curried) arguments without intervening execution.  This type should not
        /// typically used directly from either F# code or from other .NET languages.
        [<AbstractClass>]
        type FastFunc5<'T,'U,'V,'W,'X,'Y> = 
            inherit FastFunc <'T,('U -> 'V -> 'W -> 'X -> 'Y)>
            abstract member Invoke : 'T * 'U * 'V * 'W * 'X -> 'Y
            new : unit -> FastFunc5 <'T,'U,'V,'W,'X,'Y>


    //-------------------------------------------------------------------------
    // F# Mutable Reference Cells


    /// The type of mutable references.  Use the functions [:=] and [!] to get and
    /// set values of this type.
    type Ref <'a> = 
        {mutable contents: 'a;}
        /// The current value of the reference cell
        member Value : 'a with get,set
    #if CLI_AT_MOST_1_1
    #else
        /// For native interop. Pin the given object for the duration of a single call to the given function.  A native pointer to
        /// the (0,0) element in the underlying array is passed to the given function.  Resources associated with the 
        /// pin are released when the function completes, even if an exception is raised.
        [<Unverifiable>]
        member inline Pin : ('a nativeptr -> 'b) -> 'b

        /// For native interop. Pin the given object, but the caller is responsible for freeing the GCHandle
        [<Unverifiable>]
        member inline PinHandle : unit -> 'a nativeptr * System.Runtime.InteropServices.GCHandle
    #endif
            
    /// The type of mutable references.  Use the functions [:=] and [!] to get and
    /// set values of this type.
    and 'a ref = Ref<'a>


    //-------------------------------------------------------------------------
    // F# Option Types


    /// <summary>
    /// The type of optional values.  When used from other .NET languages the
    /// empty option is the 'null' value.  
    ///
    /// Use the constructors [[Some]] and [[None]] to create values of this type.
    /// Use the values in the [[Option]] module to manipulate values of this type,
    /// or pattern match against the values directly.
    /// </summary>
    ///
    /// <note>'None' values will appear as the value 'null' to other .NET languages.
    /// Instance methods on this type will appear as static methods to other .NET languages
    /// due to the use of 'null' as a value representation.
    /// </note>
    [<DefaultAugmentation(false)>]
    [<CompilationRepresentation(CompilationRepresentationFlags.PermitNull)>]
    type Option <'a> =
        /// The representation of "No value"
        | None :       'a option
        /// The representation of "Value of type 'a"
        | Some : 'a -> 'a option 
        /// Create an option value that is a 'None' value.
        static member None : 'a option
        /// Create an option value that is a 'Some' value.
        static member Some : 'a -> 'a option
        [<CompilationRepresentation(CompilationRepresentationFlags.Instance);
          Obsolete("This member has been renamed to .Value")>]
        member Item : 'a
        [<CompilationRepresentation(CompilationRepresentationFlags.Instance)>]
        /// Get the value of a 'Some' option. A NullReferenceException is raised if the option is 'None'.
        member Value : 'a
        /// Return 'true' if the option is a 'Some' value.
        member IsSome : bool
        /// Return 'true' if the option is a 'None' value.
        member IsNone : bool
  
    /// <summary>
    /// The type of optional values.  When used from other .NET languages the
    /// empty option is the 'null' value.  
    ///
    /// Use the constructors [[Some]] and [[None]] to create values of this type.
    /// Use the values in the [[Option]] module to manipulate values of this type,
    /// or pattern match against the values directly.
    /// </summary>
    ///
    /// <note>'None' values will appear as the value 'null' to other .NET languages.
    /// Instance methods on this type will appear as static methods to other .NET languages
    /// due to the use of 'null' as a value representation.
    /// </note>
    and 'a option = Option<'a>


    //-------------------------------------------------------------------------
    // F# Type Information (obsolete)


   /// The result of an F# typeof operation
    [<Obsolete("Consider using typeof<type> instead")>]
    type ReifiedType<'a> = { result: System.Type }

    //-------------------------------------------------------------------------
    // F# Compatibility Types for .NET 1.x


#if CLI_AT_MOST_1_1
namespace Microsoft.FSharp.Compatibility

    open Microsoft.FSharp.Core

    type IEnumerator<'a> =
      interface
        inherit System.Collections.IEnumerator
        inherit System.IDisposable
        abstract Current : 'a
      end
    type IEnumerable<'a> =
      interface
        inherit System.Collections.IEnumerable
        abstract GetEnumerator : unit -> IEnumerator<'a>
      end

    type KeyValuePair<'Key,'Value> =
      struct
        val Key: 'Key
        val Value: 'Value
        new : 'Key * 'Value -> KeyValuePair<'Key,'Value>
      end

    type IEqualityComparer<'key> = 
      interface 
        abstract GetHashCode : 'key -> int
        abstract Equals : 'key * 'key -> bool
      end

    type IComparer<'key> = 
      interface 
        abstract Compare: 'key * 'key -> int
      end

    type KeyNotFoundException =
        inherit System.Exception
        new : unit -> KeyNotFoundException
          
#endif


namespace Microsoft.FSharp.Core

    open System
    open Microsoft.FSharp.Core
#if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
#else
    open System.Collections.Generic
#endif

    //-------------------------------------------------------------------------
    // Triples specifying finite enumerable ranges of a (typically numeric) type
    
    [<Obsolete("This type will be removed in a future version of F#")>]
    type IRange<'a> = 
        interface
            inherit IEnumerable<'a>
            abstract Start  : 'a 
            abstract Finish : 'a option
            abstract Step   : 'a
        end

    type seq<'a> = IEnumerable<'a>


    //-------------------------------------------------------------------------
    // F# Operators


    module Operators = 

        // Arithmetic operators.  These operators are overloaded and can be used
        // on any pair of types that satisfies the constraint, e.g. the
        // '+' function can be used on any type that supports the "op_Addition" 
        // constraint.  This includes all .NET types that support the op_Addition 
        // operator.  The standard integral and floating point types support 
        // constraints as follows:
        //   - The built-in integral types are:
        //          sbyte, byte, int16, uint16, int32, unit32, 
        //          int64, uint64, nativeint, unativeint
        //
        //   - The built-in floating point types are:
        //          float, float32
        //
        //   - The built-in numeric types are these combined
        //
        //    All built-in numeric types support:
        //        'ty.(+)   : (ty,ty) -> ty
        //        'ty.(-)   : (ty,ty) -> ty
        //        'ty.( * ) : (ty,ty) -> ty
        //        'ty.(/)   : (ty,ty) -> ty
        //        'ty.(%)   : (ty,ty) -> ty
        //        'ty.(~+)  : (ty)    -> ty
        //
        //    All signed numeric types support:
        //        'ty.(~-)  : (ty)    -> ty
        /// Overloaded subtraction operator
        val inline ( - ) : (^a -> ^b -> ^c)  when (^a or ^b) : (static member ( - ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int
        /// Overloaded addition operator
        val inline ( + ) : (^a -> ^b -> ^c)  when (^a or ^b) : (static member ( + ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int
        /// Overloaded multiplication operator
        val inline ( * ) : (^a -> ^b -> ^c)  when (^a or ^b) : (static member ( * ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int
        /// Overloaded division operator
        val inline ( / ) : (^a -> ^b -> ^c)  when (^a or ^b) : (static member ( / ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int
        /// Overloaded modulo operator
        val inline ( % ) : ^a -> ^b -> ^c    when (^a or ^b) : (static member ( % ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int
        /// Overloaded logical-AND operator
        val inline (&&&): ^a -> ^a -> ^a     when ^a : (static member (&&&) : ^a * ^a    -> ^a) and default ^a : int
        /// Overloaded logical-OR operator
        val inline (|||) : ^a -> ^a -> ^a    when ^a : (static member (|||) : ^a * ^a    -> ^a) and default ^a : int
        /// Overloaded logical-XOR operator
        val inline (^^^) : ^a -> ^a -> ^a    when ^a : (static member (^^^) : ^a * ^a    -> ^a) and default ^a : int
        /// Overloaded byte-shift left operator by a specified number of bits
        val inline (<<<) : ^a -> int32 -> ^a when ^a : (static member (<<<) : ^a * int32 -> ^a) and default ^a : int
        /// Overloaded byte-shift right operator by a specified number of bits
        val inline (>>>) : ^a -> int32 -> ^a when ^a : (static member (>>>) : ^a * int32 -> ^a) and default ^a : int
        /// Overloaded logical-NOT operator
        val inline (~~~)  : ^a -> ^a         when ^a : (static member (~~~) : ^a         -> ^a) and default ^a : int
        /// Overloaded prefix=plus operator
        val inline (~+) : ^a -> ^a           when ^a : (static member (~+)  : ^a         -> ^a) and default ^a : int
        /// Structural less-than comparison
        val inline ( < ) : 'a -> 'a -> bool
        /// Structural greater-than
        val inline ( > ) : 'a -> 'a -> bool
        ///Structural greater-than-or-equal
        val inline ( >= ) : 'a -> 'a -> bool
        ///Structural less-than-or-equal comparison
        val inline ( <= ) : 'a -> 'a -> bool
        ///Structural equality
        val inline ( = ) : 'a -> 'a -> bool
        ///Structural inequality
        val inline ( <> ) : 'a -> 'a -> bool
        /// Logical right-shift of an object by a specified number of bits
        [<Obsolete("This construct is deprecated. Convert to 'uint32' and use '>>>' instead")>]
        val inline op_OverloadedLogicalRightShift: ^a -> int32 -> ^a when ^a : (static member op_RightShift : ^a * int32 -> ^a) and default ^a : int

        /// Compose two functions, the function on the left being applied first
        val inline (>>): ('a -> 'b) -> ('b -> 'c) -> ('a -> 'c) 
        /// Compose two functions, the function on the right being applied first
        val inline (<<): ('b -> 'c) -> ('a -> 'b) -> ('a -> 'c) 
        /// Apply a function to a value, the value being on the left, the function on the right
        val inline (|>): 'a -> ('a -> 'b) -> 'b
        /// Apply a function to a value, the value being on the right, the function on the left
        val inline (<|): ('a -> 'b) -> 'a -> 'b

        /// Used to specify a default value for an optional argument in the implementation of a function
        val defaultArg : 'a option -> 'a -> 'a 

        /// 1D Array element get-accessor ('getter')
        val inline ( .() ) : 'a array -> int -> 'a
        /// 1D Array element set-accessor ('setter')
        val inline ( .()<- ) : 'a array -> int -> 'a -> unit
        #if CLI_AT_LEAST_2_0
        /// 2D Array element get-accessor ('getter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline ( .(,) ) : 'a [,] -> int * int -> 'a
        /// 2D Array element set-accessor ('setter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline ( .(,)<- ) : 'a [,] -> int * int -> 'a -> unit
        /// 3D Array element get-accessor ('getter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline (.(,,))  : 'a[,,] -> int * int * int -> 'a
        /// 3D Array element get-accessor ('getter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline (.(,,)<-): 'a[,,] -> int * int * int -> 'a -> unit
        #else
        /// 2D Array element get-accessor ('getter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline ( .(,) ) : $a [,] -> int * int -> $a
        /// 2D Array element set-accessor ('setter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline ( .(,)<- ) : $a [,] -> int * int -> $a -> unit
        /// 3D Array element get-accessor ('getter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline (.(,,))  : $a[,,] -> int * int * int -> $a
        /// 3D Array element get-accessor ('getter')
        [<Obsolete("This construct is deprecated. Use 'arr.[idx1,idx2]' instead")>]
        val inline (.(,,)<-): $a[,,] -> int * int * int -> $a -> unit
        #endif
        /// The standard overloaded associative (indexed) lookup operator
        [<Obsolete("This warning is due to a lookup of the form 'obj.[idx]' where the type of 'obj' is indeterminate based on the information available at the point where the expression occurs. Consider adding further type constraints.")>]
        val inline (.[])     : ^src -> 'idx -> 'elem                           when ^src : (member get_Item   : 'idx                  -> 'elem) 
        /// The standard overloaded associative (2-indexed) lookup operator
        [<Obsolete("This warning is due to a lookup of the form 'obj.[idx,idx]' where the type of 'obj' is indeterminate based on the information available at the point where the expression occurs. Consider adding further type constraints.")>]
        val inline (.[,])    : ^src -> 'idx1 * 'idx2 -> 'elem                  when ^src : (member get_Item   : 'idx1 * 'idx2         -> 'elem)
        /// The standard overloaded associative (3-indexed) lookup operator
        [<Obsolete("This warning is due to a lookup of the form 'obj.[idx,idx,idx]' where the type of 'obj' is indeterminate based on the information available at the point where the expression occurs. Consider adding further type constraints.")>]
        val inline (.[,,])   : ^src -> 'idx1 * 'idx2 * 'idx3 -> 'elem          when ^src : (member get_Item   : 'idx1 * 'idx2 * 'idx3 -> 'elem)
        /// The standard overloaded associative (indexed) mutation operator
        [<Obsolete("This warning is due to a lookup of the form 'obj.[idx]' where the type of 'obj' is indeterminate based on the information available at the point where the expression occurs. Consider adding further type constraints.")>]
        val inline (.[]<-)   : ^src -> 'idx              -> 'elem -> unit      when ^src : (member set_Item   : 'idx                  * 'elem -> unit) 
        /// The standard overloaded associative (2-indexed) mutation operator
        [<Obsolete("This warning is due to a lookup of the form 'obj.[idx,idx]' where the type of 'obj' is indeterminate based on the information available at the point where the expression occurs. Consider adding further type constraints.")>]
        val inline (.[,]<-)  : ^src -> 'idx1 * 'idx2       -> 'elem -> unit    when ^src : (member set_Item   : 'idx1 * 'idx2         * 'elem -> unit)
        /// The standard overloaded associative (3-indexed) mutation operator
        [<Obsolete("This warning is due to a lookup of the form 'obj.[idx,idx,idx]' where the type of 'obj' is indeterminate based on the information available at the point where the expression occurs. Consider adding further type constraints.")>]
        val inline (.[,,]<-) : ^src -> 'idx1 * 'idx2 * 'idx3 -> 'elem -> unit  when ^src : (member set_Item   : 'idx1 * 'idx2 * 'idx3 * 'elem -> unit)

        /// The standard overloaded 1D-slice operator, generally used on strings, arrays and vectors. 
        /// The source collection type must support a GetSlice member. The resulting collection type must support 
        /// an Item property index.
        val inline (.[..])     : ^src -> 'idx option -> 'idx option -> ^res    when ^src : (member GetSlice : 'idx option * 'idx option -> ^res)  
                                                                               and  ^res : (member get_Item : 'idx -> 'e)
        /// The standard overloaded 2D-slice operator
        val inline (.[..,..])     : ^src -> 'idx1 option -> 'idx1 option -> 'idx2 option -> 'idx2 option -> ^res   when ^src : (member GetSlice2D : 'idx1 option * 'idx1 option * 'idx2 option * 'idx2 option -> ^res) 

        /// Raises an exception
        val inline raise : #System.Exception -> 'b 
        /// Rethrows an exception. This should only be used when handling an exception
        val inline rethrow : unit -> 'a

        /// Return the first element of a tuple, [[fst (a,b) = a]].
        val fst : 'b * 'a -> 'b
        /// Return the second element of a tuple, [[snd (a,b) = b]].
        val snd : 'a * 'b -> 'b

        /// Generic comparison
        val inline compare: 'a -> 'a -> int
        /// Maximum based on generic comparison
        val inline max : 'a -> 'a -> 'a
        /// Minimum based on generic comparison
        val inline min : 'a -> 'a -> 'a

        /// Call System.Diagnostics.Debug.Assert(condition). Calls to this function are generated when 'assert' is used as a first class function value
        val inline ``assert`` : condition:bool -> unit

        /// Ignore the passed value. This is often used to throw away results of a computation.
        val ignore : 'a -> unit

        /// Unboxes a strongly typed variable. This is the inversion of [[box]], unbox (box a) equals a.
        val inline unbox : obj -> 'a
        /// Boxes a strongly typed variable.
        val inline box : 'a -> obj

        /// Throw a 'FailureException' exception
        val failwith : string -> 'a 
        /// Throw an 'InvalidArgumentException' exception
        val invalid_arg : string -> 'a 
        /// Throw an 'InvalidOperationException' exception
        val invalid_op : string -> 'a 
        /// Throw an [[Not_found]]/[[KeyNotFoundException]] exception
        val not_found : unit -> 'a 

        /// Create a mutable reference cell
        val ref : 'a -> 'a ref
        /// Assign to a mutable reference cell
        val ( := ) : 'a ref -> 'a -> unit
        /// Dereference a mutable reference cell
        val ( ! ) : 'a ref -> 'a
        /// Negate a logical value. 'not true' equals 'false' and 'not false' equals 'true'
        val inline not : bool -> bool


        /// Returns the value of the property System.Console.Error. 
        val stderr: System.IO.TextWriter

        /// Returns the value of the property System.Console.In. 
        val stdin: System.IO.TextReader

        /// Returns the value of the property System.Console.Out. 
        val stdout: System.IO.TextWriter

        /// The standard overloaded range operator, e.g. '[n..m]' for lists, '{n..m}' for sequences
        val inline (..)    : ^a       -> ^a -> seq< ^a >    when ^a : (static member op_Range   : ^a * ^a -> seq< ^a >) and default ^a : int
        /// The standard overloaded skip range operator, e.g. '[n..m]' for lists, '{n..m}' for sequences
        val inline (.. ..) : ^a -> ^a -> ^a -> seq< ^a >    when ^a : (static member op_RangeStep   : ^a * ^a * ^a-> seq< ^a >) and default ^a : int
        
        /// Execute the function as a mutual-exlcusion region using the input value as a lock. 
        val inline lock: 'lock -> (unit -> 'a) -> 'a when 'lock : not struct

        /// Clean up resources associated with the input object after the completion of the given function.
        /// Cleanup occurs even when an exception is raised by the protected
        /// code. 
        val using: ('a :> System.IDisposable) -> ('a -> 'b) -> 'b

        /// Generate a System.Type runtime represenation of a static type.
        /// The static type is still maintained on the value returned.
        [<RequiresExplicitTypeArguments>] 
        #if CLI_AT_MOST_1_1
        val inline typeof< $a > : System.Type
        #else
        val inline typeof<'a> : System.Type
        #endif

        /// Generate a System.Type representation for a type definition. If the
        /// input type is a generic type instantiation then return the 
        /// generic type definition assocaited with all such instantiations.
        [<RequiresExplicitTypeArguments>] 
        #if CLI_AT_MOST_1_1
        #else
        val inline typedefof<'a> : System.Type
        #endif

        /// Returns the internal size of a type in bytes. For example, [[sizeof&lt;int&gt;]] returns 4.
        [<RequiresExplicitTypeArguments>] 
        #if CLI_AT_MOST_1_1
        val inline sizeof< $a > : int
        #else
        val inline sizeof<'a> : int
        #endif
        
        /// The "hash" function is a structural hash function.  It is 
        /// designed to return equal hash values for items that are 
        /// equal according to the polymorphic equality 
        /// function Pervasives.(=) (i.e. the standard "=" operator). The exact
        /// behaviour of the function can be adjusted on a type-by-type
        /// basis by implementing the IStructuralHash interface for each type.
        val inline hash: 'a -> int

        /// The "hashq" function is designed to return equal hash values
        /// for reference types that are equal according 
        /// to the physical equality '==' (see also Obj.eq). For .NET value types
        /// the implementation of Object.GetHashCode is used on a boxed version of
        /// the input.  For efficiency reasons this should not typically
        /// be used on .NET value types.  
        val inline hashq: 'a -> int

        /// Absolute value of the given number
        val inline abs      : ^a -> ^a       when ^a : (static member Abs      : ^a -> ^a)      and default ^a : int
        /// Inverse cosine of the given number
        val inline acos     : ^a -> ^a       when ^a : (static member Acos     : ^a -> ^a)      and default ^a : float
        /// Inverse sine of the given number
        val inline asin     : ^a -> ^a       when ^a : (static member Asin     : ^a -> ^a)      and default ^a : float
        /// Inverse tangent of the given number
        val inline atan     : ^a -> ^a       when ^a : (static member Atan     : ^a -> ^a)      and default ^a : float
        /// Inverse tangent of 'x/y' where 'x' and 'y' are specified separatly
        val inline atan2    : ^a -> ^a -> ^a when ^a : (static member Atan2    : ^a * ^a -> ^a) and default ^a : float
        /// Ceiling of the given number
        val inline ceil     : ^a -> ^a       when ^a : (static member Ceiling  : ^a -> ^a)      and default ^a : float
        /// Exponential of the given number
        val inline exp      : ^a -> ^a       when ^a : (static member Exp      : ^a -> ^a)      and default ^a : float

        /// Floor of the given number
        val inline floor    : ^a -> ^a       when ^a : (static member Floor    : ^a -> ^a)      and default ^a : float

        /// Sign of the given number
        val inline sign     : ^a -> int      when ^a : (static member Sign    : ^a -> int)      and default ^a : float

        /// Round the given number
        val inline round    : ^a -> ^a       when ^a : (static member Round    : ^a -> ^a)      and default ^a : float

        /// Natural logarithm of the given number
        val inline log      : ^a -> ^a       when ^a : (static member Log      : ^a -> ^a)      and default ^a : float

        /// Logarithm to base 10 of the given number
        val inline log10    : ^a -> ^a       when ^a : (static member Log10    : ^a -> ^a)      and default ^a : float

        /// Square root of the given number
        val inline sqrt     : ^a -> ^a       when ^a : (static member Sqrt     : ^a -> ^a)      and default ^a : float

        /// Cosine of the given number
        val inline cos      : ^a -> ^a       when ^a : (static member Cos      : ^a -> ^a)      and default ^a : float

        /// Hyperbolic cosine  of the given number
        val inline cosh     : ^a -> ^a       when ^a : (static member Cosh     : ^a -> ^a)      and default ^a : float
        /// Sine of the given number
        val inline sin      : ^a -> ^a       when ^a : (static member Sin      : ^a -> ^a)      and default ^a : float
        /// Hyperbolic sine of the given number
        val inline sinh     : ^a -> ^a       when ^a : (static member Sinh     : ^a -> ^a)      and default ^a : float
        /// Tangent of the given number
        val inline tan      : ^a -> ^a       when ^a : (static member Tan      : ^a -> ^a)      and default ^a : float
        /// Hyperbolic tangent of the given number
        val inline tanh     : ^a -> ^a       when ^a : (static member Tanh     : ^a -> ^a)      and default ^a : float
        //val inline truncate : ^a -> int      when ^a : (static member Truncate : ^a -> int)     and default ^a : float
        /// Overloaded power operator.
        val inline ( **  ) : ^a -> ^a -> ^a when ^a : (static member Pow : ^a * ^a -> ^a) and default ^a : float

        /// Converts the argument to byte (direct conversion for all 
        /// value types, Byte.Parse() on strings otherwise [[ToByte]] method)
        val inline byte       : ^a -> byte       when ^a : (static member ToByte    : ^a -> byte)       and default ^a : int        
        /// Converts the argument to signed byte (direct conversion for all 
        /// value types, SByte.Parse() on strings otherwise [[ToSByte]] method)
        val inline sbyte      : ^a -> sbyte      when ^a : (static member ToSByte   : ^a -> sbyte)      and default ^a : int
        /// Converts the argument to signed 16-bit integer (direct conversion for all 
        /// value types, Int16.Parse() on strings otherwise [[ToInt16]] method)
        val inline int16      : ^a -> int16      when ^a : (static member ToInt16   : ^a -> int16)      and default ^a : int
        /// Converts the argument to unsigned 16-bit integer (direct conversion for all 
        /// value types, UInt16.Parse() on strings otherwise [[ToUInt16]] method)
        val inline uint16     : ^a -> uint16     when ^a : (static member ToUInt16  : ^a -> uint16)     and default ^a : int
        /// Converts the argument to signed 32-bit integer (direct conversion for all 
        /// value types, Int32.Parse() on strings otherwise [[ToInt32]] method)
        val inline int        : ^a -> int        when ^a : (static member ToInt32   : ^a -> int)        and default ^a : int
        /// Converts the argument to a particular enum type.
        val inline enum       : int32 -> ^b        when ^b : enum<int32> 

        /// Converts the argument to signed 32-bit integer (direct conversion for all 
        /// value types, Int32.Parse() on strings otherwise [[ToInt32]] method)
        val inline int32      : ^a -> int32      when ^a : (static member ToInt32   : ^a -> int32)      and default ^a : int

        /// Converts the argument to unsigned 32-bit integer (direct conversion for all 
        /// value types, UInt32.Parse() on strings otherwise [[ToUInt32]] method)
        val inline uint32     : ^a -> uint32     when ^a : (static member ToUInt32  : ^a -> uint32)     and default ^a : int

        /// Converts the argument to signed 64-bit integer (direct conversion for all 
        /// value types, Int64.Parse() on strings otherwise [[ToInt64]] method)
        val inline int64      : ^a -> int64      when ^a : (static member ToInt64   : ^a -> int64)      and default ^a : int

        /// Converts the argument to unsigned 64-bit integer (direct conversion for all 
        /// value types, UInt64.Parse() on strings otherwise [[ToUInt64]] method)
        val inline uint64     : ^a -> uint64     when ^a : (static member ToUInt64  : ^a -> uint64)     and default ^a : int

        /// Converts the argument to 32-bit float (direct conversion for all 
        /// value types, Single.Parse() on strings otherwise [[ToSingle]] method)
        val inline float32    : ^a -> float32    when ^a : (static member ToSingle  : ^a -> float32)    and default ^a : int

        /// Converts the argument to 64-bit float (direct conversion for all 
        /// value types, Double.Parse() on strings otherwise [[ToDouble]] method)
        val inline float      : ^a -> float      when ^a : (static member ToDouble  : ^a -> float)      and default ^a : int

        /// Converts the argument to 32-bit float (direct conversion for all 
        /// value types, Single.Parse() on strings otherwise [[ToSingle]] method)
        val inline single     : ^a -> single     when ^a : (static member ToSingle  : ^a -> single)     and default ^a : int

        /// Converts the argument to 64-bit float (direct conversion for all 
        /// value types, Double.Parse() on strings otherwise [[ToDouble]] method)
        val inline double     : ^a -> float      when ^a : (static member ToDouble  : ^a -> double)     and default ^a : int

        /// Converts the argument to signed native integer (direct conversion for all 
        /// primitive numeric types and [[ToIntPtr]] method otherwise)
        val inline nativeint  : ^a -> nativeint  when ^a : (static member ToIntPtr  : ^a -> nativeint)  and default ^a : int

        /// Converts the argument to unsigned native integer using a direct conversion for all 
        /// primitive numeric types and requiring a [[ToUintPtr]] method otherwise
        val inline unativeint : ^a -> unativeint when ^a : (static member ToUIntPtr : ^a -> unativeint) and default ^a : int

        /// Converts the argument to System.Decimal using a direct conversion for all 
        /// primitive numeric types and requiring a [[ToDecimal]] method otherwise
        val inline decimal : ^a -> decimal when ^a : (static member ToDecimal : ^a -> decimal) and default ^a : int

        /// Converts the argument to character. Numeric inputs are converted according to the UTF-16 
        /// encoding for characters. String inputs must be exactly one character long.
        /// For other types a static member ToChar must exist on the type.
        val inline char        : ^a -> char      when ^a : (static member ToChar   : ^a -> char)        and default ^a : int

        //--------------------------------------------------------------------------
        // Associate other operator values with their corresponding
        // operator methods. These ones are typically associated 
        // with matrices, vectors etc.

        /// The left-oriented scalar multiplication operator, when the scalar is on the left side.
        [<Obsolete("An overload on the regular * operator should now be used instead of this operator")>]
        val inline ( $*  ) : 'a -> ^b -> ^b when ^b : (static member ( $*  )  : 'a * ^b -> ^b)

        [<Obsolete("An overload on the regular * operator should now be used instead of this operator")>]
        val inline (  *$ ) : ^a -> 'b -> ^a when ^a : (static member (  *$ )  : ^a * 'b -> ^a)

        [<Obsolete("An overload on the regular * operator should now be used instead of this operator")>]
        val inline ( $*$ ) : ^a -> ^b -> 'c when ^a : (static member ( $*$ )  : ^a * ^b -> 'c)

        [<Obsolete("An overload on the regular * operator should now be used instead of this operator")>]
        val inline ( %*  ) : ^a -> 'b -> ^a when ^a : (static member ( %*  )  : ^a * 'b -> ^a)

        [<Obsolete("An overload on the regular * operator should now be used instead of this operator")>]
        val inline (  *% ) : 'a -> ^b -> ^b when ^b : (static member (  *% )  : 'a * ^b -> ^b)

        [<Obsolete("An overload on the regular * operator should now be used instead of this operator")>]
        val inline ( %*% ) : ^a -> ^b -> 'c when ^a : (static member ( %*% )  : ^a * ^b -> 'c)

        /// Overloaded component-wise multiplication operator
        val inline ( .* ) : ^a -> 'b -> ^a when ^a : (static member ( .* )   : ^a * 'b -> ^a)
        /// Overloaded component-wise power operator
        val inline ( .^ ) : ^a -> 'b -> ^a when ^a : (static member ( .^ )   : ^a * 'b -> ^a)
        /// Overloaded component-wise division operator
        val inline ( ./ ) : ^a -> 'b -> ^a when ^a : (static member ( ./ )   : ^a * 'b -> ^a)
        /// Overloaded component-wise modulo operator
        val inline ( .% ) : ^a -> 'b -> ^a when ^a : (static member ( .% )   : ^a * 'b -> ^a)
        /// Overloaded component-wise addition operator
        val inline ( .+ ) : ^a -> 'b -> ^a when ^a : (static member ( .+ )   : ^a * 'b -> ^a)

        // In-place mutation operators for matrices etc.

        /// Overloaded in-place addition operator
        val inline (+=)  : ^a -> 'b -> unit when ^a : (static member (+=)    : ^a * ^b -> unit)
        /// Overloaded in-place subtraction operator
        val inline (-=)  : ^a -> 'b -> unit when ^a : (static member (-=)    : ^a * ^b -> unit)
        /// Overloaded in-place multiplication operator
        val inline ( *= ): ^a -> 'b -> unit when ^a : (static member ( *= )  : ^a * ^b -> unit)
        /// Overloaded in-place division operator
        val inline (/=)  : ^a -> 'b -> unit when ^a : (static member (/=)    : ^a * ^b -> unit)

        [<Obsolete("This operator has been deprecated")>]
        val inline ( *$= ) : ^a -> 'b -> unit when ^a : (static member ( *$= )   : ^a * 'b -> unit)
        [<Obsolete("This operator has been deprecated")>]
        val inline ( %*= ) : ^a -> 'b -> unit when ^a : (static member ( %*= )   : ^a * 'b -> unit)
        [<Obsolete("This operator has been deprecated")>]
        val inline ( *%= ) : 'a -> ^b -> unit when ^b : (static member ( *%= )   : 'a * ^b -> unit)

        /// Overloaded in-place component-wise multiplication operator
        [<Obsolete("This operator has been deprecated")>]
        val inline ( .*= ) : ^a -> 'b -> unit when ^a : (static member ( .*= )   : ^a * 'b -> unit)
        /// Overloaded in-place component-wise power operator
        [<Obsolete("This operator has been deprecated")>]
        val inline ( .^= ) : ^a -> 'b -> unit when ^a : (static member ( .^= )   : ^a * 'b -> unit)
        /// Overloaded in-place component-wise division operator
        [<Obsolete("This operator has been deprecated")>]
        val inline ( ./= ) : ^a -> 'b -> unit when ^a : (static member ( ./= )   : ^a * 'b -> unit)
        /// Overloaded in-place component-wise modulo operator
        [<Obsolete("This operator has been deprecated")>]
        val inline ( .%= ) : ^a -> 'b -> unit when ^a : (static member ( .%= )   : ^a * 'b -> unit)
        /// Overloaded in-place component-wise addition operator
        [<Obsolete("This operator has been deprecated")>]
        val inline ( .+= ) : ^a -> 'b -> unit when ^a : (static member ( .+= )   : ^a * 'b -> unit)
     
        module StandardRanges :
            /// Generate a range of integers  
            val int        : start:int        -> step:int        -> stop:int        -> seq<int>  
            /// Generate a range of float values
            val float      : start:float      -> step:float      -> stop:float      -> seq<float>
            /// Generate a range of float32 values
            val float32    : start:float32    -> step:float32    -> stop:float32    -> seq<float32> 
            /// Generate a range of int64 values
            val int64      : start:int64      -> step:int64      -> stop:int64      -> seq<int64> 
            /// Generate a range of uint64 values
            val uint64     : start:uint64     -> step:uint64     -> stop:uint64     -> seq<uint64> 
            /// Generate a range of uint32 values
            val uint32     : start:uint32     -> step:uint32     -> stop:uint32     -> seq<uint32> 
            /// Generate a range of nativeint values
            val nativeint  : start:nativeint  -> step:nativeint  -> stop:nativeint  -> seq<nativeint> 
            /// Generate a range of unativeint values
            val unativeint : start:unativeint -> step:unativeint -> stop:unativeint -> seq<unativeint> 
            /// Generate a range of int16 values
            val int16      : start:int16      -> step:int16      -> stop:int16      -> seq<int16> 
            /// Generate a range of uint16 values
            val uint16     : start:uint16     -> step:uint16     -> stop:uint16     -> seq<uint16> 
            /// Generate a range of sbyte values
            val sbyte      : start:sbyte      -> step:sbyte      -> stop:sbyte      -> seq<sbyte> 
            /// Generate a range of byte values
            val byte       : start:byte       -> step:byte       -> stop:byte       -> seq<byte> 
            /// Generate a range of char values
            val char       : start:char                          -> stop:char       -> seq<char> 
            /// Generate a range of values using the given zero, add, start, step and stop values
            val generate   : zero:'a -> add:('a -> 'a -> 'a) -> start:'a   -> step:'a       -> stop:'a       -> seq<'a> 


        /// This module contains the basic arithmetic operations with overflow checks.
        module Unchecked :

            /// Generate a defult value for any type. This is null for reference types, 
            /// For structs, this is struct value where all fields have the default value. 
            /// This function is unsafe in the sense that some F# values do not have proper 'null' values.
            [<RequiresExplicitTypeArguments>] 
            val inline defaultof<'a> : 'a

        /// This module contains the basic arithmetic operations with overflow checks.
        module Checked :
            /// Overloaded subtraction operator (checks for overflows)
            val inline ( - ) : (^a -> ^b -> ^c)  when (^a or ^b) : (static member ( - ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int

            /// Overloaded addition operator (checks for overflows)
            val inline ( + ) : (^a -> ^b -> ^c)  when (^a or ^b) : (static member ( + ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int

            /// Overloaded multiplication operator (checks for overflows)
            val inline ( * ) : (^a -> ^b -> ^c)  when (^a or ^b) : (static member ( * ) : ^a * ^b    -> ^c) and default ^b : ^c and default ^c : ^a and default ^c : ^b and default ^a : ^c and default ^a : ^b and default ^a : int

            /// Converts the argument to byte (direct conversion for all 
            /// primitive numeric types and [[ToByte]] method otherwise)
            val inline byte       : ^a -> byte       when ^a : (static member ToByte    : ^a -> byte)       and default ^a : int

            /// Converts the argument to signed byte (direct conversion for all 
            /// primitive numeric types and [[ToSByte]] method otherwise)
            val inline sbyte      : ^a -> sbyte      when ^a : (static member ToSByte   : ^a -> sbyte)      and default ^a : int

            /// Converts the argument to signed 16-bit integer (direct conversion for all 
            /// primitive numeric types and [[ToInt16]] method otherwise)
            val inline int16      : ^a -> int16      when ^a : (static member ToInt16   : ^a -> int16)      and default ^a : int

            /// Converts the argument to unsigned 16-bit integer (direct conversion for all 
            /// primitive numeric types and [[ToUInt16]] method otherwise)
            val inline uint16     : ^a -> uint16     when ^a : (static member ToUInt16  : ^a -> uint16)     and default ^a : int

            /// Converts the argument to signed 32-bit integer (direct conversion for all 
            /// primitive numeric types and [[ToInt32]] method otherwise)
            val inline int        : ^a -> int        when ^a : (static member ToInt32   : ^a -> int)        and default ^a : int

            /// Converts the argument to signed 32-bit integer (direct conversion for all 
            /// primitive numeric types and [[ToInt32]] method otherwise)
            val inline int32      : ^a -> int32      when ^a : (static member ToInt32   : ^a -> int32)      and default ^a : int

            /// Converts the argument to unsigned 32-bit integer (direct conversion for all 
            /// primitive numeric types and [[ToUInt32]] method otherwise)
            val inline uint32     : ^a -> uint32     when ^a : (static member ToUInt32  : ^a -> uint32)     and default ^a : int

            /// Converts the argument to signed 64-bit integer (direct conversion for all 
            /// primitive numeric types and [[ToInt64]] method otherwise)
            val inline int64      : ^a -> int64      when ^a : (static member ToInt64   : ^a -> int64)      and default ^a : int

            /// Converts the argument to unsigned 64-bit integer (direct conversion for all 
            /// primitive numeric types and [[ToUInt64]] method otherwise)
            val inline uint64     : ^a -> uint64     when ^a : (static member ToUInt64  : ^a -> uint64)     and default ^a : int

            /// Converts the argument to signed native integer (direct conversion for all 
            /// primitive numeric types and [[ToIntPtr]] method otherwise)
            val inline nativeint  : ^a -> nativeint  when ^a : (static member ToIntPtr  : ^a -> nativeint)  and default ^a : int

            /// Converts the argument to unsigned native integer (direct conversion for all 
            /// primitive numeric types and [[ToUIntPtr]] method otherwise)
            val inline unativeint : ^a -> unativeint when ^a : (static member ToUIntPtr : ^a -> unativeint) and default ^a : int

            /// Converts the argument to unicode character based on UTF16 encoding (a direct conversion for all 
            /// primitive numeric types and [[ToUIntPtr]] method otherwise)
            val inline char        : ^a -> char      when ^a : (static member ToChar   : ^a -> char)        and default ^a : int



namespace Microsoft.FSharp.Collections

#if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
#else
    open System.Collections.Generic
#endif
    open Microsoft.FSharp.Core
    /// <summary>
    /// The type of immutable singly-linked lists.  
    ///
    /// Use the constructors [] and '::' (infix) to create values of this type, or
    /// the notation [1;2;3].   Use the values in the [[List]] module to manipulate 
    /// values of this type, or pattern match against the values directly.
    /// </summary>
    ///
    type List<'a> =
        | ([])  :                               'a list
        | (::)  : _Head: 'a * _Tail: 'a list -> 'a list
        /// The module 'List' contains additional values and functions related to this type.
        [<System.Obsolete("This member is deprecated. Use 'List.Empty' instead")>]
        static member Nil : 'a list
        static member Empty : 'a list
        
        /// The module 'List' contains additional values and functions related to this type.
        member Length : int
        /// The module 'List' contains additional values and functions related to this type.
        [<System.Obsolete("This member is deprecated. Use 'IsEmpty' instead")>]
        member IsNil : bool
        member IsEmpty : bool
        /// The module 'List' contains additional values and functions related to this type.
        [<System.Obsolete("This member is deprecated. Use 'not list.IsEmpty' instead")>]
        member IsCons : bool

        /// The module 'List' contains additional values and functions related to this type,
        /// e.g. List.fold_left, List.append etc.
        member Head : 'a

        /// Build a new list whose elements are the results of applying the given function
        ///
        /// See also List.map. The module 'List' contains additional values and functions 
        /// related to this type.
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.map' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member Map : ('a -> 'b) -> 'b list

        /// Build a new collection whose elements are the results of applying the given function
        /// to each of the elements of the collection. The integer index passed to the
        /// function indicates the index (from 0) of element being transformed.
        ///
        /// See also List.map. The module 'List' contains additional values and functions 
        /// related to this type.
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.mapi' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member MapIndexed : (int -> 'a -> 'b) -> 'b list

        /// See List.iter
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.iter' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member Iterate : ('a -> unit) -> unit

        /// Return a new collection containing only the elements of the collection
        /// for which the given predicate returns "true"
        ///
        /// See also List.filter. The module 'List' contains additional values and functions 
        /// related to this type.
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.filter' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member Filter: ('a -> bool) -> 'a list

        /// Test if all elements of the collection satisfy the given predicate.
        ///
        /// See also List.for_all. The module 'List' contains additional values and functions 
        /// related to this type.
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.for_all' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member ForAll : ('a -> bool) -> bool

        /// Test if any element of the collection satisfies the given predicate.
        ///
        /// See also List.exists. The module 'List' contains additional values and functions 
        /// related to this type.
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.exists' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member Exists : ('a -> bool) -> bool

        /// Return a new list with the elements in reverse order
        ///
        /// See also List.rev. The module 'List' contains additional values and functions 
        /// related to this type.
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.rev' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member Reverse : unit -> 'a list

        /// Apply the given function to each element of the collection. The integer passed to the
        /// function indicates the index of element.
        ///
        /// See also List.iteri. The module 'List' contains additional values and functions 
        /// related to this type.
        [<System.Obsolete("This member is being deprecated in order to 'reduce the number of ways of doing the same thing' in the F# Library. Use 'List.iteri' instead, and replace the use of dot-notation with a use of the '|>' pipe operator")>]
        member IterateIndexed : (int -> 'a -> unit) -> unit

        /// The module 'List' contains additional values and functions related to this type.
        member Tail : 'a list

        /// The module 'List' contains additional values and functions related to this type.
        static member Cons : 'a * 'a list -> 'a list
        
        interface IEnumerable<'a>
        interface System.Collections.IEnumerable
        
    /// <summary>
    /// The type of immutable singly-linked lists.  
    ///
    /// Use the constructors [] and '::' (infix) to create values of this type, or
    /// the notation [1;2;3].   Use the values in the [[List]] module to manipulate 
    /// values of this type, or pattern match against the values directly.
    /// </summary>
    ///
    and 'a list = List<'a>


namespace Microsoft.FSharp.Control

    open Microsoft.FSharp.Core
    
    module LazyStatus = 
        /// This exception is raised when a lazy or other recursively referential value
        /// is used during the execution of its initialization code 
        /// The status of a lazy computation.  A future relases of F# may hid this type and thus it should not be used
        /// directly.
        type LazyStatus <'a> =
        | Delayed of (unit -> 'a)
        | Value of 'a
        | Exception of System.Exception

    exception Undefined    
    open LazyStatus
    /// The type of delayed computations.
    /// 
    /// Use the values in the [[Lazy]] module to manipulate 
    /// values of this type, and the notation 'lazy expr' to create values
    /// of this type.
    type Lazy <'a> = 
        {mutable status: LazyStatus<'a>;}
        /// The module 'Lazy' contains additional values and functions related to this type.
        member IsForced : bool
        /// The module 'Lazy' contains additional values and functions related to this type.
        member IsException : bool
        /// The module 'Lazy' contains additional values and functions related to this type.
        member IsDelayed : bool
        /// Force the execution of this value and return its result, using mutual exclusion to 
        /// prevent other threads also computing the value.
        member SynchronizedForce : unit -> 'a
        /// Force the execution of this value and return its result. Same as Value.
        member Force : unit -> 'a
        /// Force the execution of this value and return its result. Same as Value.
        member Value : 'a
        /// The module 'Lazy' contains additional values and functions related to this type.
        static member Create : (unit -> 'a) -> Lazy<'a>
    /// The type of delayed computations.
    /// 
    /// Use the values in the [[Lazy]] module to manipulate 
    /// values of this type, and the notation 'lazy expr' to create values
    /// of this type.
    and 'a ``lazy`` = Lazy<'a>

namespace Microsoft.FSharp.Control

    open Microsoft.FSharp.Core
    /// First-class listening points (i.e. objects that permit you to register a 'callback'
    /// activated when the event is triggered). See the module <tt>IEvent</tt>
    /// for functions to create events, and the type <tt>IDelegateEvent</tt>
    /// which extends this type.
    type IEvent<'a> = 
        /// Connect a listener function to the event. The listener will
        /// be invoked when the event is fired.
        abstract Add: ('a -> unit) -> unit

    /// F# gives special status to non-virtual instance member properties compatible with type IPrimitiveDelegateEvent, 
    /// generating approriate .NET metadata to make the member appear to other .NET languages as a
    /// .NET event.
    type IPrimitiveDelegateEvent<'del> (* when 'del :> System.MulticastDelegate  *) =
        /// Connect a handler delegate object to the event.  A handler can
        /// be later removed using RemoveHandler.  The listener will
        /// be invoked when the event is fired.
        abstract AddHandler: 'del -> unit
        /// Remove a listener delegate from an event listener store
        abstract RemoveHandler: 'del -> unit 

    /// .NET events are revealed to F# code as instances of the type IDelegateEvent,
    /// which is a subtype of the type [IEvent]. The delegate type parameter and 
    /// arguments are determined by the F# type checker based on information stored
    /// for the .NET event.
    ///
    /// F# code may also declare .NET events over any .NET delegate type.
    /// For example, DelegateEvent&lt;PaintEventHandler,PaintEventArgs&gt; 
    /// can be used for an event whose callbacks are of type PaintEventHandler.  The delegate
    /// and argument types must match.
    ///
    type IDelegateEvent<'del,'args> (* when 'del :> System.MulticastDelegate  *) =
        inherit IEvent<'args>
        inherit IPrimitiveDelegateEvent<'del>

    /// Fresh .NET-compatible events declared in F# are typically 
    /// values of type IHandlerEvent&lt;'a&gt;. These 
    /// hold event handlers of type Handler&lt;'a&gt;. 
    type Handler<'a> =  delegate of obj * 'a -> unit 

    /// Any instance property that has type IHandlerEvent will be published as a standard 
    /// .NET event using standard .NET delegates and metadata, and will 
    /// be recognized as an event by any language that supports the 
    /// CLI Common Language Specification. The event will accept 
    /// handlers of type Handler*lt;'a&gt;.
    type IHandlerEvent<'a> = IDelegateEvent<Handler<'a>, 'a>
    [<System.Obsolete("This type has been renamed to IHandlerEvent")>]
    type ICompatEvent<'a> = IHandlerEvent<'a>
          // member Add: ('a -> unit) -> unit
          // member AddHandler: (Handler<'a> -> unit) -> unit
          // member RemoveHandler: (Handler<'a> -> unit) -> unit

