// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 

#light

open MyLibrary.Expr
open System.Windows.Forms
open System.Drawing
open System.Drawing.Drawing2D
open System.Drawing.Imaging
open System
open System.IO
open System.ComponentModel

//---------------------------------------------------------------------------
// This next line is needed order to compile this sample on 
// both .NET v1.x and v2.0, i.e. to use CompatArray - see 
// the advanced section of the manual on '.NET Compatible Arrays when
// using .NET v1.x'
//---------------------------------------------------------------------------

open Compatibility 

//---------------------------------------------------------------------------
// This next line plugs together the parser and lexer
//---------------------------------------------------------------------------

let parseExpr text = 
    Pars.start MyLibrary.Lexer.token (Lexing.from_string text)

//---------------------------------------------------------------------------
// A function to augment the predefinedItems below with some derivatives.
// Also augment with appropriate names.
//---------------------------------------------------------------------------

let addDerivatives(expr,color) =
    [ (expr,         color, 3.0f, DashStyle.Solid  ) ;
      (Dx expr,      color, 2.0f, DashStyle.Dash   );
      (Dx(Dx(expr)), color, 1.0f, DashStyle.DashDot) ]
   
//---------------------------------------------------------------------------
// Build some test expressions for the GUI
//---------------------------------------------------------------------------

let predefinedItems = 
    [ Cube X,                 Color.SeaGreen;
      Sin X,                  Color.Blue;
      Log X,                  Color.SkyBlue;
      Exp (Poly [0.0;0.8] X), Color.HotPink;
      Prod(X,Sq (Sin X)),     Color.Red ]

//---------------------------------------------------------------------------
// Create the form and all its behaviour
//---------------------------------------------------------------------------

let form = new Form(Width = 400, Height = 400, Text = "Derivatives", Menu = new MainMenu())
  
//---------------------------------------------------------------------------
// Create the Sample menu items.  
//---------------------------------------------------------------------------

let itemsMenu = form.Menu.MenuItems.Add("&Samples")
let addOneMenuItem (expr,color) =
    let txt = MyLibrary.Expr.exprToString(expr) 
    let rec menuItem = 
        new MenuItem(txt, new EventHandler(fun _ _ -> menuItemActivate())) 
    and menuItemActivate() = 
        menuItem.Checked <- not menuItem.Checked; 
        form.Invalidate()
    itemsMenu.MenuItems.Add(menuItem) |> ignore
    menuItem

let menuItemsForPredefined =  List.map addOneMenuItem predefinedItems 
        
//---------------------------------------------------------------------------
// Create the widgets that produce the user-specified expressions, if any.  
// If a valid expression has been entered by the user it will appear 
// 'userItems'.
//---------------------------------------------------------------------------

let userItems = ref []
let userItemTextBox = new TextBox(Multiline = false)
form.Controls.Add(userItemTextBox)
userItemTextBox.TextChanged.Add(fun  _ -> 
    let text = userItemTextBox.Text
    try 
        let expr = parseExpr text
        userItems := [(expr, Color.Brown)];
        form.Invalidate()
    with _ -> 
        if !userItems <> [] then form.Invalidate();
        userItems := [])
  
//---------------------------------------------------------------------------
// The refresh function that plots the graph
//---------------------------------------------------------------------------

let guiRefresh (g: Graphics) = 
    let cliprect = form.ClientRectangle 
    let mag = 100.0 
    let minx = 0 
    let miny = 0
    let maxx = cliprect.Width - 1 
    let maxy = cliprect.Height - 1

    // Compute the origin on the grid
    let originx = cliprect.Width/2 
    let originy = cliprect.Height/2 

    // Take a float on the grid to an integer on the grid
    let limitx x = truncate(max (float(-cliprect.Width)) (min (float(cliprect.Width)) x)) 
    let limity x = truncate(max (float(-cliprect.Height)) (min (float(cliprect.Height)) x)) 
    
    // From the grid to the cartesian plane and back again
    let i2fx ix = float (ix - originx) / mag 
    let i2fy iy = float (iy - originy) / mag 
    let f2ix x = originx - limitx (mag * x) 
    let f2iy y = originy - limity (mag * y) 

    // Call the evaluator at this point.  If it fails plot at the origin.
    let evalPoint expr xi =
        let xf = i2fx xi 
        let yi = 
            try f2iy (Ex expr xf) 
            with _ -> originy
        new Point(xi, yi) 
      
    // Function to draw one expression.  Call evalPoint at each x
    // coordinate along the grid.
    let plotExpr (expr,color,width,dash) = 
        using (new Pen(color = color, DashStyle = dash, Width = width)) (fun pen -> 
            let points = CompatArray.init cliprect.Width (evalPoint expr) 
            g.DrawLines(pen,points)) 
        
    let plotExprWithDerivatives spec = 
        List.iter plotExpr (addDerivatives spec)
      
    // Draw the expressions selected from the menus
    List.iter2 
        (fun expr (menuItem:MenuItem) -> 
            if menuItem.Checked then plotExprWithDerivatives expr) 
        predefinedItems 
        menuItemsForPredefined;

    // Draw the expressions derived from the user input
    List.iter plotExprWithDerivatives !userItems;
      
    // Draw the axes and the tick marks
    using (new Pen(color=Color.Black,Width=1.0f,DashStyle=DashStyle.Solid)) (fun pen -> 
        g.DrawLine(pen, originx,0,originx,maxy); 
        g.DrawLine(pen, 0,originy,maxx,originy); 

        for xf = truncate (i2fx minx) to truncate (i2fx maxx) do
            let xi = f2ix (float xf)
            g.DrawLine(pen, xi,originy,xi,originy-6); 

        for yf = truncate (i2fy miny) to truncate (i2fy maxy) do
            let yi = f2iy (float yf)
            g.DrawLine(pen, originx,yi,originx+6,yi); 
    )

//---------------------------------------------------------------------------
// Add the paint, resize and exit logic to the form
//---------------------------------------------------------------------------

form.Closing.Add (fun _ -> Application.ExitThread())
form.Paint.Add(fun ev -> guiRefresh ev.Graphics)
form.Resize.Add(fun _ -> form.Invalidate())

//---------------------------------------------------------------------------
// Run...
//---------------------------------------------------------------------------
        
form.Visible <- true
form.Activate()
Application.Run()

 
