
//-----------------------------------------------------------------------------
// File: Matrices.fs
//
// This is a modified translation of the Managed DirectX Tutorial 3 into F#
// 
// Desc: Now that we know how to create a device and render some 2D vertices,
//       this tutorial goes the next step and renders 3D geometry. To deal with
//       3D geometry we need to introduce the use of 4x4 matrices to transform
//       the geometry with translations, rotations, scaling, and setting up our
//       camera.
//
//       Geometry is defined in model space. We can move it (translation),
//       rotate it (rotation), or stretch it (scaling) using a world transform.
//       The geometry is then said to be in world space. Next, we need to
//       position the camera, or eye point, somewhere to look at the geometry.
//       Another transform, via the view matrix, is used, to position and
//       rotate our view. With the geometry then in view space, our last
//       transform is the projection transform, which "projects" the 3D scene
//       into our 2D viewport.
//
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//-----------------------------------------------------------------------------

#light

open System  // for Colors
open System.Drawing  // for Colors
open System.Windows.Forms // for Form
open Microsoft.DirectX  // for Matrix
open Microsoft.DirectX.Direct3D // For Device, Cull etc.
open DirectXLib // see dxlib.fs

let inputData1 = 
     [| ((-1.0f,  -1.0f, 0.0f, Color.DarkGoldenrod),
         (1.0f,  -1.0f, 0.0f, Color.MediumOrchid),
         (0.0f,   1.0f, 0.0f, Color.Cornsilk)); |]

let inputData2 = 
     [| ((-1.0f,  -1.0f, 2.0f, Color.Cornsilk),
         (1.0f,  -1.0f, 2.0f, Color.MediumOrchid),
         (0.0f,   1.0f, 2.0f, Color.DarkGoldenrod)) |]

let form = NewStandardForm("F# Direct3D Tutorial 3 - Matrices",400, 300)

let device = NewStandardDevice(form)

// Turn off culling, so we see the front and back of the triangle
do device.RenderState.CullMode <- Cull.None;

// Turn off D3D lighting, since we are providing our own vertex colors
do device.RenderState.Lighting <- false 

let renderers = 
    [ Clear(Color.Blue);
      ChangingWorldTransform(fun () -> Matrix.RotationY( time()  / 150.0f ));
      View(Matrix.LookAtLH( new Vector3( 0.0f, 3.0f, -5.0f ),
                            new Vector3( 0.0f, 0.0f, 0.0f ), 
                            new Vector3( 0.0f, 1.0f, 0.0f ) ));
      Projection(Matrix.PerspectiveFovLH(PI / 4.0f, 1.0f, 1.0f, 100.0f ));
      RenderColoredPointTriangles(inputData1);
      RenderColoredPointTriangles(inputData2) ]

// Run the main code. The attribute marks the startup application thread as "Single 
// Thread Apartment" mode, which is necessary for GUI applications. 
[<STAThread>]    
do RenderInLoop(form, device, renderers)
        


