(* (c) Microsoft Corporation. All rights reserved *)
(*F# 
module Microsoft.FSharp.Compiler.Creflect
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Il = Microsoft.Research.AbstractIL.IL 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*) 

open Il
open Tast
open Tastops
open Sreflect
open Sreflect.Raw
open List
open Lib
open Printf
open Ast
open Env
open Typrelns

let imap_count m = (Imap.fold (fun a b c -> c + 1) m 0)

let mscorlibName = ""
let mkVoidTy = mkNamedTy({ tcName = "System.Void"; tcAssembly=mscorlibName},[])

type cenv = 
    { g: Env.tcGlobals; 
      amap: Import.importMap;
      scope: ccu; 
      nng: niceNameGenerator }

let mk_cenv (g,amap,scope,nng) = 
    { g = g; scope=scope; nng=nng; amap=amap } 

type env = { vs: exprVarName vspec_map; tyvs: dtypeVarIdx Imap.t }
let empty_env = { vs=vspec_map_empty(); tyvs = Imap.empty () }
let bind_typar env v = 
  let idx = (imap_count env.tyvs) in 
  { env with tyvs = Imap.add (stamp_of_typar v) idx env.tyvs }
let bind_typars env vs = fold_left bind_typar env vs (* fold_left as indexes are left-to-right *)
let bind_formal_typars env vs = bind_typars { env with tyvs=Imap.empty()} vs 
let bind_val v vR env = { env with vs = vspec_map_add v vR.vName env.vs }
let bind_vals vs vsR env = fold_right2 bind_val vs vsR env

exception InvalidQuotedTerm of exn
exception IgnoringPartOfQuotedTermWarning of string * Range.range

let wfail e = raise (InvalidQuotedTerm(e))

let recognizeFsObjModelCall expr = 
    let rec loop f args = 
        match f with 
        | TExpr_app((TExpr_val(vref,vFlags,_) as f),fty,tyargs,actualArgs,m)  when isSome(member_info_of_vref vref) ->
             Some(vref,vFlags,f,fty,tyargs,actualArgs@args)
        | TExpr_app(f,fty,[],actualArgs,m)  ->
             loop f (actualArgs @ args)
        | _ -> None in
    loop expr []
             

let rec convExpr cenv (env : env) (expr: Tast.expr) : Sreflect.Raw.expr = 
  let expr = strip_expr expr in 

  (* Recognise F# object model calls *)
  match recognizeFsObjModelCall expr with 
  | Some (vref,vFlags,f,fty,tyargs,actualArgs)  ->
        let m = range_of_expr expr in
              (* wfail(Error(Printf.sprintf "warning: error generating quotation for call to F# object member %s, nargs=%d, #actualArgs=%d" (name_of_vref vref) nargs nActualArgs,m)) *)

        let membInfo = the (member_info_of_vref vref) in
        let (numEnclTypeArgs,virtualCall,newobj,superInit,selfInit,instance,isPropGet,propset) = 
            get_member_call_info cenv.g (vref,vFlags) in

        let takesInstanceArg = instance && not newobj  in

        let nargs = (1 + (if takesInstanceArg then 1 else 0)) in 
        let nActualArgs = length actualArgs in 

        (* Too many arguments? Chop and try again *)
        if nActualArgs > nargs then 
            let now_args,later_args = chop_at nargs actualArgs in
            let rfty = apply_types (type_of_vref vref) (tyargs,now_args) in 
            convExpr cenv env (prim_mk_app (prim_mk_app (f,fty) tyargs now_args m, rfty) [] later_args m) 

        (* Right number of arguments? Work out what kind of object model call and build an object model call node. *)
        else if nActualArgs = nargs then 
            let tps,argInfos,rty,_ = dest_member_vref_typ cenv.g vref in 
            let argtys = map fst argInfos in
            let objArgs,arg = 
                match takesInstanceArg,actualArgs with 
                | false,[arg] -> [],arg
                | true,[objArg;arg] -> [objArg],arg
                | _ -> wfail(InternalError("warning: unexpected missing arguments when generating quotation for call to F# object member "^name_of_vref vref,m)) in
            let numUntupledArgs = length argInfos in 
            let untupledArgs = (if numUntupledArgs = 0 then [] else if numUntupledArgs = 1 then [arg] else try_dest_tuple arg) in 
            if length untupledArgs <> numUntupledArgs then 
                wfail(InternalError("warning: mismatch in arguments generating quotation for call to F# object member "^name_of_vref vref,m));
            let callArgs = objArgs@untupledArgs in 

            let parentTyconR = conv_tcref cenv env (actual_parent_of_vspr_vref vref) m in 
            let ctor = (newobj || superInit || selfInit) in

            (* The signature types are w.r.t. to the formal context *)
            let envinner = bind_formal_typars env tps in
            let methArgTypesR = map (conv_type cenv envinner m) argtys in
            let methRetTypeR = if is_unit_typ cenv.g rty then mkVoidTy else conv_type cenv envinner m rty in

            let methName = compiled_name_of_val (deref_val vref) in
            convObjectModelCall cenv env m tyargs (isPropGet,ctor,parentTyconR,methArgTypesR,methRetTypeR,methName,tyargs,callArgs)

        else

            (* Too few arguments? Convert to a lambda and beta-reduce the partially applied arguments to 'let' bindings *)
            let arity_info = 
               match (arity_of_vref vref) with 
               | None -> error(InternalError("no arity information found for F# object model member "^name_of_vref vref,range_of_vref vref))
               | Some a -> a in
            let expr,exprty = adjust_val_for_expected_arity cenv.g m vref vFlags arity_info in 
            convExpr cenv env (beta_mk_appl cenv.g nng (expr,exprty,[tyargs],actualArgs,m)) 
  | None -> 

  match expr with           
  (* Blast type application nodes and term application nodes apart so values are left with just their type arguments *)
  | TExpr_app(f,fty,(_ :: _ as tyargs),(_ :: _ as args),m) -> 
    let rfty = reduce_forall_typ fty tyargs in 
    convExpr cenv env (prim_mk_app (prim_mk_app (f,fty) tyargs [] m, rfty) [] args m) 

  (* Uses of polymorphic values *)
  | TExpr_app(TExpr_val(vref,isSuperInit,m),fty,tyargs,[],m2) -> 
    convValRef cenv env m vref tyargs

  (* Simple applications *)
  | TExpr_app(f,fty,tyargs,args,m) -> 
    if nonNil tyargs then wfail(Error("This quotation contains a use of a generic expression. This is not permitted",m));
    let fty = if nonNil tyargs then reduce_forall_typ fty tyargs else fty in 
    begin match args with 
    | [] -> convExpr cenv env f 
    | [h] -> mkApp(convExpr cenv env f,convExpr cenv env h)
    | h::t -> 
       let _,rfty = dest_fun_typ fty in 
       convExpr cenv env (prim_mk_app (prim_mk_app (f,fty) [] [h] m, rfty) [] t m) 
    end
  
  (* REVIEW: what is the quotation view of literals accessing enumerations? Currently they show up as integers. *)
  | TExpr_const(c,m,ty) -> conv_const cenv env m c ty
  | TExpr_val(vref,isSuperInit,m) -> convValRef cenv env m vref [] 
  | TExpr_let(bind,body,m,_) -> 
      let v = var_of_bind bind in 
      let vR = conv_val cenv env v in 
      let bodyR = convExpr cenv (bind_val v vR env) body in 
      mkLet((vR,convExpr cenv env (rhs_of_bind bind)),bodyR)
  | TExpr_letrec(binds,body,m,_) -> 
       let vs = map var_of_bind binds in 
       let vsR = map (conv_val cenv env) vs in 
       let env = (bind_vals vs vsR env) in
       let bodyR = convExpr cenv env body in 
       let bindsR = combine vsR (map (rhs_of_bind >> convExpr cenv env) binds)in 
       mkLetRec(bindsR,bodyR)
  | TExpr_lambda(_,_,vs,b,_,_,_) -> 
      let v,b = multi_lambda_to_tupled_lambda cenv.nng vs b in 
      let vR = conv_val cenv env v in 
      let bR  = convExpr cenv (bind_val v vR env) b in
      mkLambda(vR, bR)
  | TExpr_quote(raw,ast,_,_) -> 
      mkQuote(convExpr cenv empty_env ast)
  | TExpr_tlambda (_,_,_,m,_,_) -> 
      wfail(Error("Quotations may not contain contructs that are generic. Consider adding some type constraints to make this a valid quoted term",m))
  | TExpr_match (m,dtree,tgs,m2,rty,_) ->
      let typR = conv_type cenv env m rty in 
      let tgsR = 
        tgs |> Array.map (fun (TTarget(vars,e)) -> 
          let varsR = map (conv_val cenv env) vars in
          vars,
          fold_right2 
            (fun v vR acc -> mkLambda(vR,acc)) 
            vars varsR
            (convExpr cenv (bind_vals vars varsR env) e))  in 
      conv_dtree cenv env tgsR typR dtree 
         
  | TExpr_op(op,tyargs,args,m) -> 
    begin match op,tyargs,args with 
    | TOp_uconstr ucref,_,_ -> 
      let mkR = conv_ucref cenv env ucref m in 
        let typ = rty_of_uctyp ucref tyargs in
        let argtys = typs_of_ucref_rfields (mk_inst_for_stripped_typ typ) ucref in
        let tyargsR = map (conv_type cenv env m) tyargs in 
        let argsR = list_map (convExpr cenv env) args in 
        mkSum(mkR,tyargsR,argsR)
    | TOp_tuple,tyargs,_ -> 
        let tyR = conv_type cenv env m (mk_tupled_ty cenv.g tyargs) in 
        let argsR = map (convExpr cenv env) args in 
        mkTuple(tyR,argsR)
    | TOp_recd (_,tcref),_,_  -> 
        let rgtypR = conv_tcref cenv env tcref m in 
        let tyargsR = map (conv_type cenv env m) tyargs in 
        let typ = mk_tyapp_ty tcref tyargs in
        let argtys = typs_of_tcref_rfields (mk_inst_for_stripped_typ typ) tcref in 
        let argsR = map (convExpr cenv env) args in 
        mkRecdMk(rgtypR,tyargsR,argsR)
    | TOp_constr_field_get (ucref,n),tyargs,[e] -> 
        let tyargsR = map (conv_type cenv env m) tyargs in 
        let tcR,s = conv_ucref cenv env ucref m in 
        let projR = (tcR,s,n) in 
        mkSumFieldGet( projR, tyargsR,convExpr cenv env e)
    | TOp_field_get_addr(rfref),tyargs,_ -> 
        wfail(Error( "Quotations may not contain subexpressions that take the address of a field",m)) (* TODO *)
    | TOp_field_get(rfref),tyargs,[] -> 
        wfail(Error( "Quotations may not contain subexpressions that fetch static or instance fields",m)) (* TODO *)
    | TOp_field_get(rfref),tyargs,[e] -> 
        let projR = conv_rfref cenv env rfref m in 
        let tyargsR = map (conv_type cenv env m) tyargs in 
        mkRecdGet(projR,tyargsR,convExpr cenv env e)
    | TOp_tuple_field_get(n),tyargs,[e] -> 
        let tyR = conv_type cenv env m (mk_tupled_ty cenv.g tyargs) in 
        mkTupleGet(tyR, n, convExpr cenv env e)
    | TOp_asm([ Il.I_ldfld(_,_,fspec) ],_),enclTypeArgs,[objArg]  -> 
        let tyargsR = map (conv_type cenv env m) enclTypeArgs in 
        let parentTyconR = conv_il_tref cenv env (tref_of_fspec fspec) in 
        let objArg = 
            match objArg with 
            | TExpr_op(TOp_lval_op(LGetAddr,vref),[],[],m) -> expr_for_vref m vref
            | e -> e in 
        let objArgR = convExpr cenv env objArg in 
        mkFieldGet( (parentTyconR, name_of_fspec fspec),tyargsR, objArgR)
    | TOp_asm([ Il.I_arith AI_ceq ],_),_,[arg1;arg2]  -> 
        mkEquality((convExpr cenv env arg1,convExpr cenv env arg2))
    | TOp_asm _,_,_                                   	   -> wfail(Error( "Quotations may not contain subexpressions that contain .NET IL asembly code. Please consider reporting this incompleteness to fsbugs@microsoft.com",m))
    | TOp_exnconstr ecr,_,args                  	   -> wfail(Error( "Quotations may not contain subexpressions that create exception values",m))
    | TOp_field_set rfref, tinst,[rx;x]           	   -> 
        let projR = conv_rfref cenv env rfref m in 
        let tyargsR = map (conv_type cenv env m) tyargs in 
        mkRecdSet(projR,tyargsR,(convExpr cenv env rx,convExpr cenv env x))
    | TOp_constr_tag_get tycr,tinst,[cx]            	   -> wfail(Error( "Quotations may not contain subexpressions that fetch union case indexes",m))
    | TOp_constr_field_set (c,i),tinst,[cx;x]         	   -> wfail(Error( "Quotations may not contain subexpressions that set union case fields",m))
    | TOp_exnconstr_field_get(ecr,i),[],[ex]           	   -> wfail(Error( "Quotations may not contain subexpressions that fetch data from exception values",m))
    | TOp_exnconstr_field_set(ecr,i),[],[ex;x]         	   -> wfail(Error( "Quotations may not contain subexpressions that set fields in exception values",m))
    | TOp_get_ref_lval,_,_                      	   -> wfail(Error( "Quotations may not contain subexpressions that require byref pointers",m))
    | TOp_trait_call (ss),_,_   	           -> wfail(Error( "Quotations may not contain subexpressions that call trait members",m))
    | TOp_coerce,[toTy;fromTy],[x]  -> mkCoerce(conv_type cenv env m toTy,convExpr cenv env x)
    | TOp_lval_op(LGetAddr,vref),[],[] -> mkGetAddr(convExpr cenv env (TExpr_val(vref,NormalValUse,m)))
    | TOp_lval_op _,_,_ -> wfail(Error( "Quotations may not contain subexpressions that manipulate pointers",m))
    | TOp_array,[ty],xa -> mkArrayMk(conv_type cenv env m ty,(xa |> map (convExpr cenv env)))                            
    | TOp_while,[],[TExpr_lambda(_,_,[_],test,_,_,_);TExpr_lambda(_,_,[_],body,_,_,_)]  -> 
          mkEncodedWhileLoop(convExpr cenv env test, convExpr cenv env body)
    | TOp_for (true),[],[TExpr_lambda(_,_,[_],lim0,_,_,_);TExpr_lambda(_,_,[_],incrLim1,_,_,_);body]  -> 
        begin match dest_incr incrLim1 with 
        | Some(lim1) -> 
             mkEncodedForLoop(convExpr cenv env lim0,
                             convExpr cenv env lim1, 
                             convExpr cenv env body)
        | None -> wfail(Error( "Quotations may not contain descending for loops",m))
        end
    | TOp_for (false),_,_ ->
        wfail(Error( "Quotations may not contain descending for loops",m))
    | TOp_ilcall((virt,protect,valu,newobj,superInit,isPropGet,_,boxthis,mref),
                  enclTypeArgs,methTypeArgs,tys),[],callArgs -> 
         let parentTyconR = conv_il_tref cenv env (tref_of_mref mref) in 
         let ctor = (newobj || (superInit = CtorValUsedAsSuperInit) || (superInit = CtorValUsedAsSelfInit)) in
         let methArgTypesR = map (conv_il_typ cenv env m) (args_of_mref mref) in
         let methRetTypeR = conv_il_typ cenv env m (ret_of_mref mref) in
         let methName = name_of_mref mref in
         let tyargs = (enclTypeArgs@methTypeArgs) in 
         convObjectModelCall cenv env m tyargs (isPropGet,ctor,parentTyconR,methArgTypesR,methRetTypeR,methName,tyargs,callArgs)
    | TOp_try_finally,[resty],[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[_],e2,_,_,_)] -> 
        warning(IgnoringPartOfQuotedTermWarning( "ignoring finally block in quotation term",m));
        convExpr cenv env e1
    | TOp_try_catch,[resty],[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[vf],ef,_,_,_); TExpr_lambda(_,_,[vh],eh,_,_,_)] -> 
        warning(IgnoringPartOfQuotedTermWarning( "ignoring catch handler in quotation term",m));
        convExpr cenv env e1
    | TOp_bytes _,_,_ -> 
        wfail(Error( "Quotations may not contain literal byte arrays",m))
    | _ -> 
        wfail(InternalError( "unexpected expression shape",m))

(*     | _ -> wfail(Error( "Quotations may not contain subexpressions",m)) *)
  end
  | TExpr_hole(m,ty)  -> mkHole(conv_type cenv env m ty)
  | TExpr_seq (x0,x1,NormalSeq,m)  -> mkSeq(convExpr cenv env x0, convExpr cenv env x1)
  | TExpr_obj (_,typ,_,_,[TMethod(TSlotSig(_,ctyp, _,_,_,_),tps,tmvs,e,_) as tmethod],_,m,_) when is_delegate_typ typ -> 
       let f = mk_lambdas m tps tmvs (e,rty_of_tmethod tmethod) in
       let fR = convExpr cenv env f in 
       let tyargR = conv_type cenv env m ctyp in 
       mkDelegate(tyargR, fR)

   | TExpr_static_optimization (tcs,csx,x,m)           	   -> convExpr cenv env x
   | TExpr_tchoose _  -> convExpr cenv env (Typrelns.choose_typar_solutions_for_tchoose cenv.g cenv.amap expr)
   | TExpr_seq  (x0,x1,ThenDoSeq,m)                   	   -> mkSeq(convExpr cenv env x0, convExpr cenv env x1)
   | TExpr_obj (n,typ,basev,basecall,overrides,iimpls,m,_)      -> wfail(Error( "unhandled construct in AST: TExpr_obj",m))
   | TExpr_link r -> convExpr cenv env !r


and convObjectModelCall cenv env m tyargs (isPropGet,ctor,parentTyconR,methArgTypesR,methRetTypeR,methName,tyargs,callArgs) =

    let tyargsR = map (conv_type cenv env m) tyargs in 
    let callArgsR = map (convExpr cenv env) callArgs in 

    match isPropGet,ctor,callArgs with 
     | true,_,[objArg] -> 
         let objArg = 
             match objArg with 
             (* This clause is needed for structs *)
             | TExpr_op(TOp_lval_op(LGetAddr,vref),[],[],m) -> expr_for_vref m vref
             | e -> e in 
         let objArgR = convExpr cenv env objArg in 
         mkPropGet( (parentTyconR, chop_property_name methName),tyargsR, objArgR)
     | _,true,_ ->
       mkCtorCall( { ctorParent   = parentTyconR; 
                     ctorArgTypes = methArgTypesR },
                   tyargsR, callArgsR)
     | _,_,_ ->
       mkMethodCall( { methParent   = parentTyconR; 
                       methArgTypes = methArgTypesR;
                       methRetType  = methRetTypeR;
                       methName     = methName },
                     tyargsR, callArgsR)


and convValRef cenv env m vref tyargs =
      let v = deref_val vref in 
      if vspec_map_mem v env.vs then (
          if nonNil tyargs then wfail(InternalError("ignoring generic application of local quoted variable",m));
          mkVar(vspec_map_find v env.vs)
      ) else (
        let vty = (type_of_val v) in
        match vref with 
        | Ref_private v -> 
            (* We allow quotations to contain references to values private within modules as *)
            (* long as the path to the values is stable. *)
            (match pubpath_of_val v with 
             | None -> wfail(Error("Quotation of a private value. Quotations may not contain references to private values",m))
             | Some pubpath -> 
                let nlpath = enclosing_nlpath_of_pubpath cenv.scope pubpath in 
                conv_nlpath_val cenv env m nlpath (name_of_val v) tyargs vty)
        | Ref_nonlocal nlref ->
          let nlpath = nlpath_of_nlref nlref in
          conv_nlpath_val cenv env m nlpath (item_of_nlref nlref) tyargs vty
      )

and conv_nlpath_val cenv env m nlpath nm tyargs vty =
    let assref = (match qualified_name_of_ccu (ccu_of_nlpath nlpath) with Some s -> s | None -> ".") in 
    let tyargsR = map (conv_type cenv env m) tyargs in 
    let vtyR = 
        let tps,vtau = try_dest_forall_typ vty in 
        let env = bind_typars empty_env tps in 
        conv_type cenv env m vtau in 
    let vrefR = { topDefAssembly=assref;
                  topDefPath=(path_of_nlpath nlpath,nm);
                  topDefTypeScheme = vtyR } in 
    mkAnyTopDefn(vrefR,tyargsR)
and conv_ucref cenv env ucref m =
    let ucgtypR = conv_tcref cenv env (tcref_of_ucref ucref) m in 
    (ucgtypR,name_of_ucref ucref) 
and conv_rfref cenv env rfref m =
    let typR = conv_tcref cenv env (tcref_of_rfref rfref) m in 
    (typR,name_of_rfref rfref)
and conv_val cenv env v = 
  let ty = conv_type cenv env (range_of_val v) (type_of_val v) in
  { vName = conv_val_name cenv env v ty; }
and conv_typar (cenv:cenv) (env:env) tp = 
  { tvName = name_of_typar tp }
and conv_tpref cenv env m tp = 
  if Imap.mem (stamp_of_tpref tp) env.tyvs then begin
    Imap.find (stamp_of_tpref tp) env.tyvs 
  end else
    wfail(InternalError("Quotations may not be generic",m))
and conv_val_name cenv env v ty = 
  freshExprVarName (name_of_val v^"#"^string_of_int (stamp_of_val v)) ty
and conv_type cenv env m typ =
  match strip_tpeqns_and_tcabbrevs typ with 
  | TType_app(tcref,[tyarg]) when is_il_arr_tcref cenv.g tcref -> 
      mkArrayTy(rank_of_il_arr_tcref cenv.g tcref,
                   conv_type cenv env m tyarg)
  | TType_app(tcref,tyargs) -> 
      mkNamedTy(conv_tcref cenv env tcref m, map (conv_type cenv env m)  tyargs)
  | TType_fun(a,b)          -> mkFunTy(conv_type cenv env m a,conv_type cenv env m b)
  | TType_tuple(l)          -> conv_type cenv env m (compiled_tuple_ty cenv.g l)
  | TType_var(tp)           -> mkVarTy(conv_tpref cenv env m tp)
  | TType_forall(spec,ty)   -> wfail(Error("Generic functions are not permitted in quoted terms. Consider adding some type constraints.",m))
  | _ -> wfail(InternalError ("Quotations may not contain this kind of type",m))

and conv_const cenv env m c ty =
  match try_elim_bigint_bignum_constants cenv.g m c with 
  | Some e -> convExpr cenv env e
  | None ->
      match c with 
      | TConst_bool    i ->  mkBool i
      | TConst_int8    i ->  mkSByte i
      | TConst_uint8   i ->  mkByte i
      | TConst_int16   i ->  mkInt16 i
      | TConst_uint16  i ->  mkUInt16 i
      | TConst_int32   i ->  mkInt32 i
      | TConst_uint32  i ->  mkUInt32 i
      | TConst_int64   i ->  mkInt64 i
      | TConst_uint64  i ->  mkUInt64 i
      | TConst_float   i ->  mkDouble(Nums.ieee64_to_float i)
      | TConst_float32 i ->  mkSingle((*F# Nums.ieee32_to_float32 F#*) i)
      | TConst_string  s ->  mkString(Bytes.unicode_bytes_as_string s)
      | TConst_char    c ->  mkChar c
      | TConst_unit      ->  mkUnit()
      | TConst_default     ->  mkNull(conv_type cenv env m ty)
      | _ -> wfail(Error ("Quotations may not contain this kind of constant",m))


and conv_dtree cenv env tgsR typR x = 
  match x with 
  | TDSwitch(e1,csl,dfltOpt,m) -> 
    let e1R = convExpr cenv env e1 in 
    fold_right 
      (fun (TCase(discrim,dtree)) acc -> 
        match discrim with 
        | TTest_unionconstr(ucref,tyargs) -> 
          let ucR = conv_ucref cenv env ucref m in 
          let tyargsR = map (conv_type cenv env m) tyargs in 
          mkCond(mkSumTagTest(ucR,tyargsR,e1R),conv_dtree cenv env tgsR typR dtree,acc)
        | TTest_const(TConst_bool(true)) -> 
          mkCond(e1R,conv_dtree cenv env tgsR typR dtree,acc)
        | TTest_const(TConst_bool(false)) -> 
          mkCond(e1R,acc,conv_dtree cenv env tgsR typR dtree)
        | TTest_const(c) -> 
          let ty = type_of_expr cenv.g e1 in
          mkCond(mkEquality((conv_const cenv env m c (type_of_expr cenv.g e1),e1R)),conv_dtree cenv env tgsR typR dtree,acc)
        | TTest_isnull -> 
          mkCond(mkEquality((conv_const cenv env m TConst_default (type_of_expr cenv.g e1),e1R)),conv_dtree cenv env tgsR typR dtree,acc)
        | TTest_isinst (srcty,tgty) -> wfail(Error( "Quotations may not contain type tests",m))
        | TTest_query _ -> wfail(InternalError( "TTest_query test in quoted term",m))
        | TTest_array_length _ -> wfail(Error( "Quotations may not contain array pattern matching",m))
       )
       csl
       (match dfltOpt with 
        | Some d -> conv_dtree cenv env tgsR typR d 
        | None -> wfail(Error( "Quotations may not contain this kind of pattern match",m)))
  | TDSuccess (es,n) -> 
     let vs,tg = tgsR.(n) in 
     fold_left2 (fun acc v e -> mkApp(acc,convExpr cenv env e)) tg vs es 
  | TDBind(bind,rest) -> 
      let v = var_of_bind bind in 
      let vR = conv_val cenv env v in 
      let envinner = (bind_val v vR env) in 
      let bodyR = conv_dtree cenv envinner tgsR typR rest  in 
      mkLet((vR,convExpr cenv env (rhs_of_bind bind)),bodyR)


      (* REVIEW: quotation references to items in the assembly being generated *)
      (* are persisted as assembly-qualified-name strings. However this means *)
      (* they are not correctly re-adjusted *)
      (* when static-linking. We should probably instead persist these references *)
      (* by creating fake IL metadata (e.g. fields) that refer to the relevant assemblies, which then *)
      (* get fixed-up automatically by IL metadata rewriting. *)
and full_tname_of_tref tr = String.concat "+" (tr.trefNested @ [tr.trefName])

and conv_il_tref cenv env tr = 
  let tname = full_tname_of_tref tr in 
  let scoref = scoref_of_tref tr in 
  let assref = 
     (* We use the name "." as a special marker for the "local" assembly *)
      match scoref with 
      | ScopeRef_local -> "."
      | _ -> qualified_name_of_scoref scoref  in 
  {tcName = tname; tcAssembly = assref}
  
and conv_il_typ cenv env m ty = 
  match ty with 
  | Type_boxed tspec | Type_value tspec -> 
    mkNamedTy(conv_il_tref cenv env (tref_of_tspec tspec), 
              map (conv_il_typ cenv env m) (inst_of_tspec tspec))
  | Type_array (shape,ty) -> 
    mkArrayTy(Nums.i32_to_int (rank_of_array_shape shape),conv_il_typ cenv env m ty)
  | Type_tyvar idx -> mkVarTy(Nums.u16_to_int idx)
  | Type_void -> mkVoidTy
  | Type_ptr _ 
  | Type_byref _ 
  | Type_modified _ 
  | Type_other _ | Type_fptr _ ->  wfail(Error( "Quotations may not contain this kind of type",m))
  
  
and conv_tcref cenv env tcref m = 
  let repr = il_repr_of_tcref tcref in 
  match repr with 
  | TyrepOpen asm -> wfail(Error( "Quotations may not contain this kind of type",m))
  | TyrepNamed (tref,boxity) -> 
      match tcref with 
      | Ref_private _ -> 
         (* We use the name "." as a special marker for the "local" assembly *)
         { tcName= full_tname_of_tref tref; tcAssembly="." }
      | _ -> conv_il_tref cenv env tref
 
