// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Compatibility.OCaml

open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
open System
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

/// <summary>
///   Immutable maps using structural comparison
/// </summary>
///
/// <performance> Maps based on structural comparison are  
/// efficient. They are not a suitable choice if keys are recursive data structures 
/// or require non-structural comparison semantics.</performance>
module Map = 

    //--------------------------------------------------------------------------
    // Map.Make functor
    //
    // Immutable maps using user-defined comparison.

    /// A collection of operations for creating and using maps based on a particular comparison function.
    /// The 'tag type parameter is used to track information about the comparison function.
    type Provider<'key,'a,'tag> 
     when 'tag :> IComparer<'key> =
        { empty: Tagged.Map<'key,'a,'tag>;
          add: 'key -> 'a -> Tagged.Map<'key,'a,'tag> -> Tagged.Map<'key,'a,'tag>;
          find: 'key -> Tagged.Map<'key,'a,'tag> -> 'a;
          first: 'b. ('key -> 'a -> 'b option) -> Tagged.Map<'key,'a,'tag> -> 'b option;
          tryfind: 'key -> Tagged.Map<'key,'a,'tag> -> 'a option;
          remove: 'key -> Tagged.Map<'key,'a,'tag> -> Tagged.Map<'key,'a,'tag>;
          mem: 'key -> Tagged.Map<'key,'a,'tag> -> bool;
          iter: ('key -> 'a -> unit) -> Tagged.Map<'key,'a,'tag> -> unit;
          map:  'b. ('a -> 'b) -> Tagged.Map<'key,'a,'tag> -> Tagged.Map<'key,'b,'tag>;
          mapi: 'b. ('key -> 'a -> 'b) -> Tagged.Map<'key,'a,'tag> -> Tagged.Map<'key,'b,'tag>;
          fold: 'b. ('key -> 'a -> 'b -> 'b) -> Tagged.Map<'key,'a,'tag> -> 'b -> 'b }

    type Provider<'key,'a> = Provider<'key,'a,IComparer<'key>>
    
    val Make: ('key -> 'key -> int) -> Provider<'key,'a>

    /// A functor to build a collection of operations for creating and using 
    /// maps based on the given comparison function. This returns a record that 
    /// contains the functions you use to create and manipulate maps of
    /// this kind.  The returned value is much like an ML module. 
    ///
    /// Language restrictions related to polymorphism may mean you
    /// have to create a new instantiation of for each toplevel
    /// key/value type pair.
    ///
    /// To use this function you need to define a new named class that implements IComparer and
    /// pass an instance of that class as the first argument. For example:
    ///      type MyComparer = 
    ///          new() = { }
    ///          interface IComparer&lt;string&gt; with 
    ///            member self.Compare(x,y) = ...
    ///
    /// let MyStringMapProvider : Map.Provider &lt; string,int &gt; = Map.MakeTagged(new MyComparer())
    val MakeTagged: ('tag :> IComparer<'key>) -> Provider<'key,'a,'tag>

    [<Obsolete("Consider using Tagged.Map<'key,'a> instead")>]
    type CMap<'key,'a> = Tagged.Map<'key,'a>

    [<Obsolete("Consider using Map.Provider<'key,'a> instead")>]
    type CMapOps<'key,'a> = Provider<'key,'a>

