(* (c) Microsoft Corporation. All rights reserved *)

(*F#

module Microsoft.FSharp.Compiler.Sreflect
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal
open Microsoft.Research.AbstractIL.Internal.Nums
open Microsoft.FSharp.Compiler
open Microsoft.FSharp.Compiler.Lib
F#*)

(*IF-OCAML*) 
open Lib
open Nums

type float32 =  ieee32 
type sbyte = i8
type byte = u8
type int16 = i16
type uint16 = u16
type int32 = i32
type uint32 = u32
type int64 = i64
type uint64 = u64
type char = unichar
(*ENDIF-OCAML*)

module Bytebuf = Bytes.Bytebuf
module Bytestream = Bytes.Bytestream

(*---------------------------------------------------------------------------
 * Expression and Type Specifications.  These are what we pickle.
 *------------------------------------------------------------------------- *)

open Nums

module Raw = struct

    type ('a,'b) family  = 'a -> 'b

    let mkRLinear mk (vs,body) = List.fold_left (fun acc v -> mk (v,acc)) body (List.rev vs) 

    (*F# 
    open Microsoft.FSharp.Text.StructuredFormat.LayoutOps
    F#*)

    type dtype =
      | VarType   of dtypeVarIdx
      | AppType   of dtycon * dtype list
      
    and dtypeVarIdx = int
    and dtypeVar = { tvName: string; }
    and dtycon = 
    | ArrayTyOp  of int (* rank *) 
    | FunTyOp
    | NamedTyOp of dtyconNamed
    and dtyconNamed = { tcName: string; tcAssembly:  string }
    let mkVarTy v = VarType v 
    let mkFunTy (x1,x2) = AppType(FunTyOp, [x1; x2])  
    let mkArrayTy (n,x) = AppType(ArrayTyOp n, [x]) 
    let mkNamedTy (r,l) = AppType(NamedTyOp r,l) 


    (** Represents specifications of a subset of F# expressions *)
    type expr =
      | CombExpr   of exprConstr * dtype list * expr list
      | VarExpr    of exprVarName
      | QuoteExpr  of expr 
      | LambdaExpr of exprVar * expr 
      | HoleExpr   of dtype 
      
    and exprVarName = { vText: string; vType: dtype } 

    and exprVar = { vName: exprVarName }
    and exprConstr = 
    | AppOp
    | CondOp  
    | TopDefOp of topDefData
    | LetRecOp  
    | LetRecCombOp  
    | LetOp  
    | RecdMkOp  of dtyconNamed
    | RecdGetOp  of dtyconNamed * string
    | RecdSetOp  of dtyconNamed * string
    | SumMkOp  of dtyconNamed * string
    | SumFieldGetOp  of dtyconNamed * string * int
    | SumTagTestOp of dtyconNamed * string
    | TupleMkOp  
    | TupleGetOp of int
    | EqualityOp 
    | UnitOp   
    | BoolOp   of bool   
    | StringOp of string 
    | SingleOp of float32 
    | DoubleOp of float  
    | CharOp   of unichar   
    | SByteOp  of sbyte
    | ByteOp   of byte   
    | Int16Op  of int16  
    | UInt16Op of uint16 
    | Int32Op  of int32  
    | UInt32Op of uint32 
    | Int64Op  of int64  
    | UInt64Op of uint64 
    (* Object Model Goop! *)
    | PropGetOp of dtyconNamed * string
    | FieldGetOp of dtyconNamed * string
    | CtorCallOp of ctorSpec 
    | MethodCallOp of methSpec 
    | CoerceOp 
    | ArrayMkOp
    | DelegateOp
    (* Unclean/imperative! *)
    | GetAddrOp 
    | SeqOp 
    | EncodedForLoopOp 
    | EncodedWhileLoopOp 
    | NullOp   

    and topDefPath = (string list * string)
    and topDefData = { topDefAssembly: string; topDefPath: topDefPath; topDefTypeScheme: dtype }
    and ctorSpec = { ctorParent: dtyconNamed; ctorArgTypes: dtype list }
    and methSpec = { methParent: dtyconNamed; methName: string; methArgTypes: dtype list ; methRetType: dtype }
      
    type 'a opFamily = ('a, exprConstr) family

    type op = unit opFamily

    type expr2 = (expr * expr)
    type expr3 = (expr * expr * expr)
    type exprN = expr list

    type 'a exprFamily = ('a, expr) family
                                 
    type ('a      ) exprFamilyGN1 = ('a   * dtype list * expr ) exprFamily
    type ('a      ) exprFamilyGN2 = ('a   * dtype list * expr2) exprFamily
    type ('a      ) exprFamilyGNN = ('a   * dtype list * exprN) exprFamily
                                                                            
    let mkVar v = VarExpr v 
    let mkHole v = HoleExpr v 
    let mkApp (a,b) = CombExpr(AppOp, [], [a; b]) (* REVIEW: type arguments? *)
    let mkLambda (a,b) = LambdaExpr (a,b) 
    let mkQuote (a) = QuoteExpr (a) 
    let mkGN0 op (r,tyargs,())          = CombExpr (op r,tyargs, [])
    let mkGN1 op (r,tyargs,x)           = CombExpr (op r,tyargs, [x]) 
    let mkGN2 op (r,tyargs,(x1,x2))     = CombExpr (op r,tyargs, [x1;x2]) 
    let mkGN3 op (r,tyargs,(x1,x2,x3))  = CombExpr (op r,tyargs, [x1;x2;x3]) 
    let mkGNN op (r,tyargs,l)           = CombExpr (op r,tyargs, l)

    let mkFN0 op (tyargs,())          = CombExpr (op, tyargs, []) 
    let mkFN1 op (tyargs,x)           = CombExpr (op, tyargs, [x]) 
    let mkFN2 op (tyargs,(x1,x2))     = CombExpr (op, tyargs, [x1; x2]) 
    let mkFN3 op (tyargs,(x1,x2,x3))  = CombExpr (op, tyargs, [x1; x2; x3]) 
    let mkFNN op (tyargs,l)           = CombExpr (op, tyargs, l) 

    let mkGNG_to_mkGEG mk (r,args) = mk (r,[],args) 
    let mkGG0_to_mkGGE mk (r,args) = mk (r,args,()) 
      
    let mkGNG_to_mkG1G mk (r,ty,args) = mk (r,[ty],args) 
    let mkGNG_to_mkG2G mk (r,(ty1,ty2),args) = mk (r,[ty1;ty2],args) 
    let mkGE0_to_mkGEE mk r = mk(r,()) 
    let mk0GG_to_mkEGG mk (b,c) = mk((),b,c) 
    let mk0EG_to_mkEEG mk (c) = mk((),c)

    let mkGEE op x = mkGE0_to_mkGEE (mkGNG_to_mkGEG (mkGN0 op)) x
    let mkGE1 op x =                 mkGNG_to_mkGEG (mkGN1 op) x
    let mkGE2 op x =                 mkGNG_to_mkGEG (mkGN2 op) x
    let mkGE3 op x =                 mkGNG_to_mkGEG (mkGN3 op) x
    let mkGEN op x =                 mkGNG_to_mkGEG (mkGNN op) x
    let mkG11 op x =                 mkGNG_to_mkG1G (mkGN1 op) x
    let mkGNE op x =                 mkGG0_to_mkGGE (mkGN0 op) x
    let mkE11 op x =                 mk0GG_to_mkEGG (mkG11 op) x
    let mkEE1 op x =                 mk0EG_to_mkEEG (mkGE1 op) x
    let mkEE2 op x =                 mk0EG_to_mkEEG (mkGE2 op) x
    let mkEE3 op x =                 mk0EG_to_mkEEG (mkGE3 op) x
    let mkEEN op x =                 mk0EG_to_mkEEG (mkGEN op) x

    let op v () = v
    
    let mkCond x = mkEE3 (op CondOp) x
    let mkAnyTopDefn x = mkGNE (fun x -> TopDefOp x) x
    let mkTuple (ty,x) =  mkFNN TupleMkOp  ([ty],x)
    let mkLetRaw x = mkEE2 (op LetOp) x
    let mkLet ((v,e),b) = mkLetRaw (e,mkLambda (v,b))  (* nb. order preserves source order *)
    let mkUnit () = CombExpr (UnitOp, [], []) 
    let mkNull ty = CombExpr (NullOp, [ty], []) 

    let mkLetRecRaw x = mkEE2 (op LetRecOp) x
    let mkLetRecCombRaw x = mkEEN (op LetRecCombOp) x
    let mkLetRec (ves,body) = 
         let vs,es = List.split ves in 
         let vtys = vs |> List.map (fun x -> x.vName.vType) in 
         mkLetRecRaw
         (mkRLinear mkLambda  (vs, mkLetRecCombRaw es),
          mkRLinear mkLambda (vs,body))
          
    let mkRecdMk    e = e |> mkGNN (fun x -> RecdMkOp x)
    let mkRecdGet   e = e |> mkGN1 (fun (x,y) -> RecdGetOp (x,y))
    let mkRecdSet   e = e |> mkGN2 (fun (x,y) -> RecdSetOp (x,y))
    let mkSum   e = e |> mkGNN (fun (x,y) -> SumMkOp (x,y))
    let mkSumFieldGet  e = e |> mkGN1 (fun (x,y,z) -> SumFieldGetOp (x,y,z))
    let mkSumTagTest e = e |> mkGN1 (fun (x,y) -> SumTagTestOp (x,y))
    let mkTupleGet  (ty,n,e) = (n,ty,e) |> mkG11 (fun x -> TupleGetOp x)

    let mkEquality (e1,e2) = mkEE2 (op EqualityOp) (e1,e2)
    let mkCoerce e = e |> (mkGN1 (op CoerceOp) |> mkGNG_to_mkG1G |> mk0GG_to_mkEGG)
    let mkArrayMk e = e |> (mkGNN (op ArrayMkOp) |> mkGNG_to_mkG1G |> mk0GG_to_mkEGG)

    let mkBool   e = e |> mkGEE ((fun x -> BoolOp x))
    let mkString e = e |> mkGEE ((fun x -> StringOp x))
    let mkSingle e = e |> mkGEE ((fun x -> SingleOp x))
    let mkDouble e = e |> mkGEE ((fun x -> DoubleOp x))
    let mkChar   e = e |> mkGEE ((fun x -> CharOp x))
    let mkSByte  e = e |> mkGEE ((fun x -> SByteOp x))
    let mkByte   e = e |> mkGEE ((fun x -> ByteOp x))
    let mkInt16  e = e |> mkGEE ((fun x -> Int16Op x))
    let mkUInt16 e = e |> mkGEE ((fun x -> UInt16Op x))
    let mkInt32  e = e |> mkGEE ((fun x -> Int32Op x))
    let mkUInt32 e = e |> mkGEE ((fun x -> UInt32Op x))
    let mkInt64  e = e |> mkGEE ((fun x -> Int64Op x))
    let mkUInt64 e = e |> mkGEE ((fun x -> UInt64Op x))

    let mkGetAddr    e = e |> mkEE1 (op GetAddrOp)
    let mkSeq        e = e |> mkEE2 (op SeqOp)
    let mkEncodedForLoop    e = e |> mkEE3 (op EncodedForLoopOp)
    let mkEncodedWhileLoop  e = e |> mkEE2 (op EncodedWhileLoopOp)
    let mkDelegate e = e |> mkE11 (op DelegateOp)
    let mkPropGet e = e |> mkGN1 (fun (x,y) -> PropGetOp (x,y))
    let mkFieldGet e = e |> mkGN1 (fun (x,y) -> FieldGetOp (x,y))
    let mkCtorCall e = e |> mkGNN (fun x -> CtorCallOp x)
    let mkMethodCall e = e |> mkGNN (fun x -> MethodCallOp x)

    (*---------------------------------------------------------------------------
     * Pickle/unpickle expression and type specifications
     *------------------------------------------------------------------------- *)

    let pickledDefinitionsResourceNameBase = "PickledDefinitions"

    let freshExprVarName = 
      let i = ref 0 in 
      fun n ty -> i := !i + 1; {vText=(n^"."^string_of_int (!i)); vType=ty}

    module SimplePickle = struct

        type 'a tbl = 
            { tbl: ('a, int) Hashtbl.t;
              mutable rows: 'a list;
              mutable count: int }

        let new_tbl () = 
          { tbl = (Hashtbl.create 20);
            rows=[];
            count=0; }

        let get_tbl tbl = List.rev tbl.rows
        let tbl_size tbl = List.length tbl.rows

        let add_entry tbl x =
          let n = tbl.count in 
          tbl.count <- tbl.count + 1;
          Hashtbl.add tbl.tbl x n;
          tbl.rows <- x :: tbl.rows;
          n

        let find_or_add_entry tbl x =
          if Hashtbl.mem tbl.tbl x then Hashtbl.find tbl.tbl x 
          else add_entry tbl x


        let tbl_find tbl x = Hashtbl.find tbl.tbl x 

        let tbl_mem tbl x = Hashtbl.mem tbl.tbl x 

        type outstate = 
          { os: Bytebuf.t; 
            ostrings: string tbl }

        let pbyte b st = Bytebuf.emit_int_as_byte st.os b

        type 'a pickler = 'a -> outstate -> unit

        let pbool b st = pbyte (if b then 1 else 0) st
        let p_void (os: outstate) = ()
        let p_unit () (os: outstate) = ()
        let prim_pint32 i st = 
          pbyte (b0 i) st;
          pbyte (b1 i) st;
          pbyte (b2 i) st;
          pbyte (b3 i) st

        (* compress integers according to the same scheme used by CLR metadata *)
        (* This halves the size of pickled data *)
        let pint32 n st = 
          if n >= 0l &&  n <= 0x7Fl then 
            pbyte (b0 n) st
          else if n >= 0x80l && n <= 0x3FFFl then  begin
            pbyte (??? (0x80l ||| (n lsr 8))) st; 
            pbyte (??? (n &&& 0xFFl)) st 
          end else begin
            pbyte 0xFF st;
            prim_pint32 n st
          end

        let pbytes s st = 
          let len = Bytes.length s in
          pint32 (Int32.of_int len) st;
          Bytebuf.emit_bytes st.os s

        let prim_pstring s st = 
            let bytes = Bytes.string_as_utf8_bytes s in
            let len = Bytes.length bytes in
            pint32 (Int32.of_int len) st;
            Bytebuf.emit_bytes st.os bytes

        let p_int c st = pint32 (Int32.of_int c) st
        let pint8 i st = pint32 (Nums.i8_to_i32 i) st
        let puint8 i st = pbyte (Nums.u8_to_int i) st
        let pint16 i st = pint32 (Nums.i16_to_i32 i) st
        let p_uint16 x st = pint32 (Nums.u16_to_i32 x) st
        let p_uint32 x st = pint32 (Nums.u32_to_i32 x) st
        let pint64 i st = 
          pint32 (Int64.to_int32 (Int64.logand i 0xFFFFFFFFL)) st;
          pint32 (Int64.to_int32 (Int64.shift_right_logical i 32)) st
        let puint64 x st = pint64 (Nums.u64_to_i64 x) st
        let pfloat64 i st = pint64 (Int64.bits_of_float i) st
        let pfloat32 i st = pint32 (Nums.ieee32_to_bits i) st
        let punichar i st = p_uint16 (Nums.unichar_to_u16 i) st
        let (*F# inline F#*) p_tup2 p1 p2 (a,b) (st:outstate) = (p1 a st : unit); (p2 b st : unit)
        let (*F# inline F#*) p_tup3 p1 p2 p3 (a,b,c) st = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit)
        let (*F# inline F#*) p_tup4 p1 p2 p3 p4 (a,b,c,d) st = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit)
        let puniq tbl key st = p_int (find_or_add_entry tbl key) st
        let p_string s st = puniq st.ostrings s st
        let rec p_list f x st =
          match x with 
          | [] -> pbyte 0 st
          | h :: t -> pbyte 1 st; f h st; p_list f t st
              

        let pickle_obj p x =
          let stringTab,phase1bytes =
            let st1 = 
              { os = Bytebuf.create 100000; 
                ostrings=new_tbl();
                 } in
            p x st1;
            get_tbl st1.ostrings, Bytebuf.close st1.os in
          let phase2data = (stringTab,phase1bytes) in 
          let phase2bytes = 
            let st2 = 
             { os = Bytebuf.create 100000; 
                ostrings=new_tbl();} in
            p_tup2 (p_list prim_pstring) pbytes phase2data st2;
            Bytebuf.close st2.os in 
          phase2bytes
    end

    open SimplePickle
    let rec p_expr x st =
      match x with 
      | CombExpr(c,ts,args) -> pbyte 0 st; p_tup3 pconstSpec (p_list pdtype) (p_list p_expr) (c,ts,args) st
      | VarExpr v           -> pbyte 1 st; pexprVarName v st
      | LambdaExpr(v,e)     -> pbyte 2 st; p_tup2 pvarSpec p_expr (v,e) st
      | HoleExpr(ty)        -> pbyte 3 st; pdtype ty st
      | QuoteExpr(tm)       -> pbyte 4 st; p_expr tm st
    and pexprVarName v st   = p_tup2 p_string pdtype (v.vText,v.vType) st
    and pvarSpec v st       = pexprVarName v.vName st
    and precdFieldSpec v st = p_tup2 pnamedTyconstSpec p_string v st
    and puconstrSpec v st   = p_tup2 pnamedTyconstSpec p_string v st
    and pdtype x st =
      match x with 
      | VarType v     -> pbyte 0 st; p_int v st
      | AppType(c,ts) -> pbyte 1 st; p_tup2 ptyconstSpec (p_list pdtype) (c,ts) st
    and ptyvarSpec v st = p_string v.tvName st
    and ptyconstSpec x st = 
      match x with
      | FunTyOp     -> pbyte 1 st
      | NamedTyOp a -> pbyte 2 st; pnamedTyconstSpec a st
      | ArrayTyOp a -> pbyte 3 st; p_int a st
    and pconstSpec x st = 
      match x with 
      | CondOp        -> pbyte 0 st
      | TopDefOp a    -> pbyte 1 st; ptopDefData a st
      | LetRecOp      -> pbyte 2 st
      | RecdMkOp  a   -> pbyte 3 st; pnamedTyconstSpec a st
      | RecdGetOp  (x,y)  -> pbyte 4 st; precdFieldSpec (x,y) st
      | SumMkOp  (x,y)  -> pbyte 5 st; puconstrSpec (x,y) st
      | SumFieldGetOp (a,b,c) -> pbyte 6 st; p_tup2 puconstrSpec p_int ((a,b),c) st
      | SumTagTestOp (x,y) -> pbyte 7 st; puconstrSpec (x,y) st
      | TupleMkOp     -> pbyte 8 st
      | TupleGetOp a  -> pbyte 9 st; p_int a st
      | EqualityOp    -> pbyte 10 st
      | BoolOp   a    -> pbyte 11 st; pbool a st
      | StringOp a    -> pbyte 12 st; p_string a st
      | SingleOp a    -> pbyte 13 st; pfloat32 a st
      | DoubleOp a    -> pbyte 14 st; pfloat64 a st
      | CharOp   a    -> pbyte 15 st; punichar a st
      | SByteOp  a    -> pbyte 16 st; pint8 a st
      | ByteOp   a    -> pbyte 17 st; puint8 a st
      | Int16Op  a    -> pbyte 18 st; pint16 a st
      | UInt16Op a    -> pbyte 19 st; p_uint16 a st
      | Int32Op  a    -> pbyte 20 st; pint32 a st
      | UInt32Op a    -> pbyte 21 st; p_uint32 a st
      | Int64Op  a    -> pbyte 22 st; pint64 a st
      | UInt64Op a    -> pbyte 23 st; puint64 a st
      | UnitOp        -> pbyte 24 st
      | PropGetOp (a,b)   -> pbyte 25 st; p_tup2 pnamedTyconstSpec p_string (a,b) st
      | CtorCallOp a  -> pbyte 26 st; p_tup2 pnamedTyconstSpec (p_list pdtype) (a.ctorParent,a.ctorArgTypes) st
      | GetAddrOp     -> pbyte 27 st
      | CoerceOp      -> pbyte 28 st
      | SeqOp         -> pbyte 29 st
      | EncodedForLoopOp     -> pbyte 30 st
      | MethodCallOp a -> pbyte 31 st; p_tup4 pnamedTyconstSpec (p_list pdtype) pdtype p_string (a.methParent,a.methArgTypes,a.methRetType, a.methName) st
      | ArrayMkOp      -> pbyte 32 st
      | DelegateOp     -> pbyte 33 st
      | EncodedWhileLoopOp -> pbyte 34 st
      | LetOp            -> pbyte 35 st
      | RecdSetOp  (x,y) -> pbyte 36 st; precdFieldSpec (x,y) st
      | FieldGetOp (a,b) -> pbyte 37 st; p_tup2 pnamedTyconstSpec p_string (a, b) st
      | LetRecCombOp     -> pbyte 38 st
      | AppOp            -> pbyte 39 st
      | NullOp        -> pbyte 40 st
      
    and pnamedTyconstSpec x st = p_tup2 p_string passref (x.tcName, x.tcAssembly) st
    and passref x st = p_string x st
    and ptopDefPath a st = p_tup2 (p_list p_string) p_string a st
    and ptopDefData a st = p_tup3 passref ptopDefPath pdtype (a.topDefAssembly, a.topDefPath, a.topDefTypeScheme) st
    let pickle = pickle_obj p_expr
    let pdmkn = p_tup4 ptopDefPath (p_list ptyvarSpec) pdtype p_expr
    let pdmkns = p_list pdmkn
    let pickleDefns   = pickle_obj pdmkns
end (* end of Raw module *)

