/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Collections;
using System.Data;
using System.Data.Common;
using System.Globalization;
using System.Reflection;
using System.ComponentModel;

using PostgreSql.Data.NPgClient;
using PostgreSql.Data.PgSqlClient.DbSchema;
using PostgreSql.Data.PgTypes;

namespace PostgreSql.Data.PgSqlClient
{	
	public sealed class PgDataReader : MarshalByRefObject, IEnumerable, IDataReader, IDisposable, IDataRecord
	{		
		#region Fields
		
		private const int		STARTPOS = -1;
		private bool			disposed;
		private bool			open;
		private int				position;
		private int				recordsAffected;
		private int				fieldCount;
		private DataTable		schemaTable;
		private CommandBehavior	behavior;
		private PgCommand		command;
		private PgConnection	connection;
		private object[]		row;

		#endregion

		#region Constructors

		private PgDataReader()
		{
			this.open				= true;		
			this.position			= STARTPOS;
			this.recordsAffected	= -1;						
			this.fieldCount			= -1;
		}

		internal PgDataReader(PgCommand command, PgConnection connection) : this()
		{
			this.command	= command;
			this.behavior	= this.command.CommandBehavior;
			this.connection	= connection;
		}

		#endregion

		#region Finalizer

		~PgDataReader()
		{
			this.Dispose(false);
		}

		#endregion

		#region IDisposable Methods

		void IDisposable.Dispose() 
		{
			this.Dispose(true);
			System.GC.SuppressFinalize(this);
		}

		private void Dispose(bool disposing)
		{
			if (!this.disposed)
			{
				try
				{
					if (disposing)
					{
						// release any managed resources
						this.Close();

						this.command	= null;
						this.connection	= null;
						this.row		= null;
						this.schemaTable= null;
					}

					// release any unmanaged resources
				}
				finally
				{
				}
							
				this.disposed = true;
			}
		}

		#endregion

		#region IDataReader Properties & Methods

		public int Depth 
		{
			get { return 0; }
		}

		public bool IsClosed
		{
			get { return !open; }
		}

		public int RecordsAffected 
		{
			get { return IsClosed ? recordsAffected : -1; }
		}

		public bool HasRows
		{
			get 
			{ 
				return command.Statement.HasRows;
			}
		}

		public void Close()
		{
			if (!this.open)
			{
				return;
			}

			// This will update RecordsAffected property
			this.updateRecordsAffected();

			if (this.command != null && !this.command.IsDisposed)
			{
				if (this.command.Statement != null)
				{
					// Set values of output parameters
					this.command.InternalSetOutputParameters();		
				}
			}

			if (this.connection != null)
			{
				this.connection.DataReader = null;

				if ((this.behavior & CommandBehavior.CloseConnection) == CommandBehavior.CloseConnection)
				{
					this.connection.Close();
				}
			}

			this.open		= false;			
			this.position	= STARTPOS;
		}

		public bool NextResult()
		{
			if (this.IsClosed)
			{
				throw new InvalidOperationException("The datareader must be opened.");
			}

			this.updateRecordsAffected();

			bool returnValue = this.command.NextResult();

			if (returnValue)
			{
				this.fieldCount	= this.command.Statement.RowDescriptor.Fields.Length;
				this.position	= STARTPOS;
			}
			else
			{
				this.row = null;
			}

			return returnValue;
		}

		public bool Read()
		{
			bool read = false;

			if ((this.behavior == CommandBehavior.SingleRow && 
				this.position != STARTPOS) || 
				!this.command.Statement.HasRows)
			{
			}
			else
			{
				try
				{
					this.fieldCount = this.command.Statement.RowDescriptor.Fields.Length;

					this.position++;

					row = command.Statement.FetchRow();

					read = (this.row == null) ? false : true;
				}
				catch (PgClientException ex)
				{
					throw new PgException(ex.Message, ex);
				}
			}
						
			return read;
		}

		#endregion

		#region GetSchemaTable Method

		public DataTable GetSchemaTable()
		{
			if (schemaTable == null)
			{				
				schemaTable = getSchemaTableStructure();

				schemaTable.BeginLoadData();

				PgCommand cInfoCmd	= getColumnInfoCmd();
				PgCommand pKeyCmd	= getPrimaryKeyInfoCmd();

				for (int i = 0; i < command.Statement.RowDescriptor.Fields.Length; i++)
				{
					object[]		columnInfo	= null;
					System.Array	pKeyInfo	= null;

					// Execute commands
					cInfoCmd.Parameters[0].Value = command.Statement.RowDescriptor.Fields[i].OidNumber;
					cInfoCmd.Parameters[1].Value = command.Statement.RowDescriptor.Fields[i].OidTable;

					pKeyCmd.Parameters[0].Value	 = command.Statement.RowDescriptor.Fields[i].OidTable;

					cInfoCmd.InternalPrepare();
					pKeyCmd.InternalPrepare();

					cInfoCmd.InternalExecute();					
					pKeyCmd.InternalExecute();

					// Get Column Information
					if (cInfoCmd.Statement.Rows != null		&&
						cInfoCmd.Statement.Rows.Length > 0)
					{
						columnInfo = (object[])cInfoCmd.Statement.Rows[0];
					}

					// Get Primary Key Info
					if (pKeyCmd.Statement.Rows != null		&&
						pKeyCmd.Statement.Rows.Length > 0)
					{
						object[] temp = (object[])pKeyCmd.Statement.Rows[0];
						pKeyInfo = (System.Array)temp[3];
					}

					// Add row information
					DataRow schemaRow = schemaTable.NewRow();					

					schemaRow["ColumnName"]			= getName(i);
					schemaRow["ColumnOrdinal"]		= i;
					schemaRow["ColumnSize"]			= getSize(i);
					if (isNumeric(i))
					{
						schemaRow["NumericPrecision"]	= getNumericPrecision(i);
						schemaRow["NumericScale"]		= getNumericScale(i);
					}
					else
					{
						schemaRow["NumericPrecision"]	= DBNull.Value;
						schemaRow["NumericScale"]		= DBNull.Value;
					}
					schemaRow["DataType"]			= getFieldType(i);
					schemaRow["ProviderType"]		= getProviderType(i);
					schemaRow["IsLong"]				= isLong(i);
					schemaRow["IsRowVersion"]		= false;
					schemaRow["IsUnique"]			= false;
					schemaRow["IsAliased"]			= isAliased(i);
					schemaRow["IsExpression"]		= isExpression(i);
					schemaRow["BaseCatalogName"]	= System.DBNull.Value;
					if (columnInfo != null)
					{
						schemaRow["IsReadOnly"]			= (bool)columnInfo[10];
						schemaRow["IsAutoIncrement"]	= (bool)columnInfo[10];
						schemaRow["IsKey"]				= isPrimaryKey(pKeyInfo, (short)columnInfo[6]);
						schemaRow["AllowDBNull"]		= ((bool)columnInfo[9]) ? false : true;
						schemaRow["BaseSchemaName"]		= columnInfo[0].ToString();						
						schemaRow["BaseTableName"]		= columnInfo[1].ToString();
						schemaRow["BaseColumnName"]		= columnInfo[2].ToString();
					}
					else
					{
						schemaRow["IsReadOnly"]			= false;
						schemaRow["IsAutoIncrement"]	= false;
						schemaRow["IsKey"]				= false;
						schemaRow["AllowDBNull"]		= System.DBNull.Value;						
						schemaRow["BaseSchemaName"]		= System.DBNull.Value;
						schemaRow["BaseTableName"]		= System.DBNull.Value;
						schemaRow["BaseColumnName"]		= System.DBNull.Value;
					}

					schemaTable.Rows.Add(schemaRow);
				}

				schemaTable.EndLoadData();

				cInfoCmd.Dispose();
				pKeyCmd.Dispose();
			}			

			return schemaTable;
		}

		private PgCommand getColumnInfoCmd()
		{
			IDbSchema dbSchema = PgDbSchemaFactory.GetSchema(PgDbSchemaType.Columns);
					
			dbSchema.AddWhereFilter("pg_attribute.attnum = @OidNumber");
			dbSchema.AddWhereFilter("pg_attribute.attrelid = @OidTable");
			
			PgCommand cmd = new PgCommand(dbSchema.GetCommandText(null), command.Connection);

			cmd.Parameters.Add("@OidNumber", PgDbType.Int4);
			cmd.Parameters.Add("@OidTable", PgDbType.Int4);
	
			return cmd;
		}

		private PgCommand getPrimaryKeyInfoCmd()
		{
			IDbSchema dbSchema = PgDbSchemaFactory.GetSchema(PgDbSchemaType.Primary_Keys);
					
			dbSchema.AddWhereFilter("pg_class.oid = @OidTable");
					
			PgCommand cmd = new PgCommand(dbSchema.GetCommandText(null), command.Connection);

			cmd.Parameters.Add("@OidTable", PgDbType.Int4);

			return cmd;
		}

		private bool isPrimaryKey(System.Array pKeyInfo, short ordinal)
		{
			if (pKeyInfo != null)
			{
				for (int i = pKeyInfo.GetLowerBound(0); i <= pKeyInfo.GetUpperBound(0); i++)
				{
					if ((short)pKeyInfo.GetValue(i) == ordinal)
					{
						return true;
					}
				}
			}

			return false;
		}

		#endregion

		#region Indexers

		public object this[int i]
		{
			get { return GetValue(i); }
		}

		public object this[string name]
		{			
			get { return GetValue(GetOrdinal(name)); }
		}

		#endregion

		#region IDataRecord Properties % Methods
		
		public int FieldCount
		{			
			get { return command.Statement.RowDescriptor.Fields.Length; }
		}

		public String GetName(int i)
		{
			return getName(i);
		}

		[EditorBrowsable(EditorBrowsableState.Never)]
		public String GetDataTypeName(int i)
		{
			return getDataTypeName(i);
		}

		public Type GetFieldType(int i)
		{			
			return getFieldType(i);
		}

		public object GetValue(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return getValue(i);
		}

		public int GetValues(object[] values)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}
			
			for (int i = 0; i < FieldCount; i++)
			{
				values[i] = GetValue(i);
			}

			return values.Length;
		}

		public int GetOrdinal(string name)
		{
			if (IsClosed)
			{
				throw new InvalidOperationException("Reader closed");
			}

			return getOrdinal(name);
		}

		public bool GetBoolean(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToBoolean(GetValue(i));
		}

		public byte GetByte(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToByte(GetValue(i));
		}

		public long GetBytes(int i, long dataIndex, byte[] buffer, 
							int bufferIndex, int length)
		{
			return 0;
		}

		[EditorBrowsable(EditorBrowsableState.Never)]
		public char GetChar(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}
			
			return Convert.ToChar(GetValue(i));
		}

		public long GetChars(int i, long dataIndex, char[] buffer, 
							int bufferIndex, int length)
		{
			return 0;
		}
		
		public Guid GetGuid(int i)
		{
			return new Guid();
		}

		public Int16 GetInt16(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToInt16(GetValue(i));
		}

		public Int32 GetInt32(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToInt32(GetValue(i));
		}
		
		public Int64 GetInt64(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToInt64(GetValue(i));
		}

		public float GetFloat(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToSingle(GetValue(i));
		}

		public double GetDouble(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}
			
			return Convert.ToDouble(GetValue(i));
		}

		public String GetString(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToString(GetValue(i));
		}

		public Decimal GetDecimal(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToDecimal(GetValue(i));
		}

		public DateTime GetDateTime(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return Convert.ToDateTime(GetValue(i));
		}

		public TimeSpan GetTimeSpan(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return (TimeSpan)GetValue(i);
		}

		public PgPoint GetPgPoint(int i)
		{
			return (PgPoint)this.GetPgValue(i);
		}

		public PgBox GetPgBox(int i)
		{
			return (PgBox)this.GetPgValue(i);
		}

		public PgLSeg GetPgLSeg(int i)
		{
			return (PgLSeg)this.GetPgValue(i);
		}

		public PgCircle GetPgCircle(int i)
		{
			return (PgCircle)this.GetPgValue(i);
		}

		public PgPath GetPgPath(int i)
		{
			return (PgPath)this.GetPgValue(i);
		}

		public PgPolygon GetPgPolygon(int i)
		{
			return (PgPolygon)this.GetPgValue(i);
		}

		public PgTimeSpan GetPgTimeSpan(int i)
		{
			return new PgTimeSpan(this.GetTimeSpan(i));
		}

		public object GetPgValue(int i)
		{
			switch (this.getProviderType(i))
			{
				case PgDbType.Interval:
					return this.GetPgTimeSpan(i);

				default:
					return this.getValue(i);
			}
		}

		public int GetPgValues(object[] values)
		{
			return this.GetValues(values);
		}

		public IDataReader GetData(int i)
		{
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			return getData(i);
		}		

		public bool IsDBNull(int i)
		{	
			if (position == STARTPOS)
			{
				throw new InvalidOperationException("There are no data to read.");
			}

			if (i < 0 || i >= FieldCount)
			{
				throw new IndexOutOfRangeException("Could not find specified column in results.");
			}
						
			return isDBNull(i);
		}

		#endregion

		#region IEnumerable Methods

		IEnumerator IEnumerable.GetEnumerator()
		{
			return new DbEnumerator(this);
		}

		#endregion

		#region Private Methods

		private IDataReader getData(int i)
		{
			return null;
		}

		private int getOrdinal(string name)
		{
			for (int i = 0; i < command.Statement.RowDescriptor.Fields.Length; i++)
			{
				if (cultureAwareCompare(command.Statement.RowDescriptor.Fields[i].FieldName, name))
				{
					return i;
				}
			}

			return -1;
		}

		private string getName(int i)
		{
			return command.Statement.RowDescriptor.Fields[i].FieldName;
		}

		private int getSize(int i)
		{
			return command.Statement.RowDescriptor.Fields[i].DataType.Size;			
		}

		private int getNumericPrecision(int i)
		{
			#warning "Add implementation"
			return 0;
		}

		private int getNumericScale(int i)
		{
			#warning "Add implementation"
			return 0;
		}

		private Type getFieldType(int i)
		{
			return command.Statement.RowDescriptor.Fields[i].DataType.SystemType;
		}

		private PgDbType getProviderType(int i)
		{
			return (PgDbType)command.Statement.RowDescriptor.Fields[i].DataType.DataType;
		}

		private string getDataTypeName(int i)
		{
			return command.Statement.RowDescriptor.Fields[i].DataType.Name;
		}

		private object getValue(int i)
		{
			return row[i];
		}

		private bool isNumeric(int i)
		{				
			if (i < 0 || i >= FieldCount)
			{
				throw new IndexOutOfRangeException("Could not find specified column in results.");
			}
			
			return command.Statement.RowDescriptor.Fields[i].DataType.IsNumeric();
		}

		private bool isLong(int i)
		{			
			if (i < 0 || i >= FieldCount)
			{
				throw new IndexOutOfRangeException("Could not find specified column in results.");
			}

			return command.Statement.RowDescriptor.Fields[i].DataType.IsLong();
		}

		private bool isDBNull(int i)
		{
			bool returnValue = false;

			if (row[i] == System.DBNull.Value)
			{
				returnValue = true;
			}

			return returnValue;
		}

		private bool isAliased(int i)
		{
			/* TODO: Add implementation	*/
			return false;
		}

		private bool isExpression(int i)
		{	
			bool returnValue = false;

			if (command.Statement.RowDescriptor.Fields[i].OidNumber == 0 &&
				command.Statement.RowDescriptor.Fields[i].OidTable	== 0)
			{
				returnValue = true;
			}

			return returnValue;
		}

		private DataTable getSchemaTableStructure()
		{
			DataTable schema = new DataTable("Schema");			

			schema.Columns.Add("ColumnName"		, System.Type.GetType("System.String"));
			schema.Columns.Add("ColumnOrdinal"	, System.Type.GetType("System.Int32"));
			schema.Columns.Add("ColumnSize"		, System.Type.GetType("System.Int32"));
			schema.Columns.Add("NumericPrecision", System.Type.GetType("System.Int32"));
			schema.Columns.Add("NumericScale"	, System.Type.GetType("System.Int32"));
			schema.Columns.Add("DataType"		, System.Type.GetType("System.Type"));
			schema.Columns.Add("ProviderType"	, System.Type.GetType("System.Int32"));
			schema.Columns.Add("IsLong"			, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("AllowDBNull"	, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("IsReadOnly"		, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("IsRowVersion"	, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("IsUnique"		, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("IsKey"			, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("IsAutoIncrement", System.Type.GetType("System.Boolean"));
			schema.Columns.Add("IsAliased"		, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("IsExpression"	, System.Type.GetType("System.Boolean"));
			schema.Columns.Add("BaseSchemaName"	, System.Type.GetType("System.String"));
			schema.Columns.Add("BaseCatalogName", System.Type.GetType("System.String"));
			schema.Columns.Add("BaseTableName"	, System.Type.GetType("System.String"));
			schema.Columns.Add("BaseColumnName"	, System.Type.GetType("System.String"));
			
			return schema;
		}

		private void updateRecordsAffected()
		{
			if (this.command != null && !this.command.IsDisposed)
			{
				if (this.command.RecordsAffected != -1)
				{
					this.recordsAffected = 
						this.recordsAffected == -1 ? 0 : this.recordsAffected;
					this.recordsAffected += this.command.RecordsAffected;
				}
			}
		}

		private bool cultureAwareCompare(string strA, string strB)
		{
			return CultureInfo.CurrentCulture.CompareInfo.Compare(
				strA, 
				strB, 
				CompareOptions.IgnoreKanaType | CompareOptions.IgnoreWidth | 
				CompareOptions.IgnoreCase) == 0 ? true : false;
		}

		#endregion
	}
}