/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.atbean;

import static org.junit.Assert.*;

import static org.springframework.config.java.util.BeanMetadata.COPY_HIDDEN;

import org.junit.Test;

import org.springframework.beans.factory.config.BeanDefinition;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.annotation.Meta;
import org.springframework.config.java.context.JavaConfigApplicationContext;
import org.springframework.config.java.internal.model.ConfigurationModelBeanDefinitionReaderTests;

import test.common.beans.ITestBean;
import test.common.beans.TestBean;


/** TODO: JAVADOC */
public class BeanAnnotationTests {

    /** Bean methods should be 'inherited'. */
    @Test
    public void testNewAnnotationNotRequiredOnConcreteMethod() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(InheritsWithoutNewAnnotation.class);

        TestBean tom = ctx.getBean(TestBean.class, "tom");
        TestBean becky = ctx.getBean(TestBean.class, "becky");
        assertSame(tom, becky.getSpouse());
        assertSame(becky, ctx.getBean("becky"));
    }


    @Test
    public void testBeanMetadata() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(MetaConfig.class);
        BeanDefinition beanDefinition = ctx.getBeanFactory().getBeanDefinition("testBean");
        assertNotNull("attribute was not set", beanDefinition.getAttribute(COPY_HIDDEN));
        assertEquals("incorrect attribute value", "true", beanDefinition.getAttribute(COPY_HIDDEN));
    }


    /**
     * @see  ConfigurationModelBeanDefinitionReaderTests#renderWithAliases() (unit test)
     */
    @Test
    public void testBeanAliases() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(AliasesConfiguration.class);

        ITestBean aliasedBean = ctx.getBean(ITestBean.class, "aliasedBean");
        assertEquals("Legion", aliasedBean.getName());
        assertSame(aliasedBean, ctx.getBean("aliasedBean"));
        assertSame(aliasedBean, ctx.getBean("tom"));
        assertSame(aliasedBean, ctx.getBean("dick"));
        assertSame(aliasedBean, ctx.getBean("harry"));
        assertFalse(ctx.containsBean("Glen"));
    }

    @Configuration
    public abstract static class DefinesAbstractBeanMethod {
        @Bean
        public TestBean becky() {
            TestBean becky = new TestBean();
            becky.setSpouse(tom());
            return becky;
        }

        @Bean
        public abstract TestBean tom();
    }

    public static class InheritsWithoutNewAnnotation extends DefinesAbstractBeanMethod {
        @Override
        public TestBean tom() { return new TestBean(); }
    }

    @Configuration
    public static class MetaConfig {
        @Bean(meta = @Meta(key = COPY_HIDDEN, value = "true"))
        public TestBean testBean() { return new TestBean(); }
    }

    @Configuration
    public static class AliasesConfiguration {
        @Bean(aliases = { "tom", "dick", "harry" })
        public TestBean aliasedBean() {
            TestBean tb = new TestBean();
            tb.setName("Legion");
            return tb;
        }
    }

}
