/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.support;

import static java.lang.String.format;

import org.springframework.beans.FatalBeanException;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.config.java.annotation.Bean;


/**
 * Unchecked exception thrown by {@link ConfigurationSupport#getObject(FactoryBean)}
 * in the event that the underlying {@link FactoryBean#getObject()} method throws
 * an exception.  It is important that {@code ConfigurationSupport}'s
 * {@code getObject} method does not propagate checked exceptions, as it would
 * make usage cumbersome within {@link Bean @Bean} methods.
 * <p/>
 * The original {@code FactoryBean} exception is simply wrapped and propagated
 * along as the root cause.
 *
 * @author Chris Beams
 */
@SuppressWarnings("serial")
public class FactoryBeanObjectCreationException extends FatalBeanException {

    /**
     * Create a new {@link FactoryBeanObjectCreationException} with the
     * specified root cause.
     * @param factoryBean the {@link FactoryBean} that threw the exception
     * @param cause the root cause
     */
    public FactoryBeanObjectCreationException(FactoryBean factoryBean, Exception cause) {
        super(format("FactoryBean %s failed during call to getObject() " +
                "with exception message: %s", factoryBean, cause.getMessage()), cause);
    }

    /**
     * Create a new {@link FactoryBeanObjectCreationException} with the
     * specified <var>message</var> explaining the failure.
     *
     * @param message reason for failure
     */
    public FactoryBeanObjectCreationException(String message) {
        super(message);
    }

}
